package jp.ecuacion.splib.web.service;

import java.util.Optional;
import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.splib.jpa.repository.SplibRepository;
import org.springframework.orm.ObjectOptimisticLockingFailureException;

public interface SplibJpaServiceInterface<E extends AbstractEntity> {

  // /** findAndOptimisticLockingCheck() にて使用。監査項目（最終更新者・最終更新日時）をnullに設定。 */
  // →全く不要だったと発覚。。。
  // public abstract void putNullToAuditingFields(E e);

  /** findAndOptimisticLockingCheck() にて使用。selectのためのrepositoryを取得。 */
  public abstract SplibRepository<E, Long> getRepositoryForOptimisticLocking();

  /** findAndOptimisticLockingCheck() にて使用。楽観的排他制御のためのversionを取得。 */
  public abstract String getVersion(E e);

  /**
   * 一覧画面から単票画面への遷移・編集時・削除時などで、楽観的排他制御チェックを行う処理。併せてDBからentityを取得する。
   * 本メソッド内で、あらかじめ画面表示の際に取得しhtml内に埋めたversionと別途本処理の中で取得したentityのversion比較を行う。
   * 現時点での差異確認は行うが、その後目的を達成するまでには、もう一度楽観的チェックが行われるので、必ず本メソッドの戻り値として取得するentityを後続で使用すること。
   * （たとえばデータupdate時は、本チェックでversionに際がないことを確認した後、update処理実施までに他のユーザからupdateされた場合もエラーとする必要があるため。）
   * <p>
   * 比較するversionは、1テーブルの場合はversionを普通に格納すれば良いのだが、複数テーブルを一度に更新する場合などは、そのそれぞれのテーブルのversionを比較する必要が出る。
   * その場合、画面上では「3,5」のように複数の値を持ち、DBから取得するersionの値も同じ形にしてstringで比較を行う。
   * 尚、どのテーブルを更新するか否かに関わらず、どれかひとつでもversionにずれがあればエラーとする仕様。
   * そもそも頻度が高いわけではないので、他のテーブルだろうと更新があったら確認し直してから行更新してくれ、と。
   * </p>
   */
  public default AbstractEntity findAndOptimisticLockingCheck(String idOfRootRecord,
      String versionInScreen) throws AppException {
    Optional<E> optional =
        getRepositoryForOptimisticLocking().findById(Long.valueOf(idOfRootRecord));

    // 他のユーザがレコードを削除してしまい、削除フラグが立って検索できない状態になる場合がある。
    // それを見越して、指定のIDのデータが存在しない場合も排他制御エラーとする。
    if (optional.isEmpty()) {
      String msg = "jp.ecuacion.splib.web.common.message.sameRecordAlreadyDeleted";
      throw new CustomizedValidationAppException(msg);
    }

    E e = optional.get();

    // versionが異なる場合は楽観的排他制御エラーとする
    if (!versionInScreen.equals(getVersion(e))) {
      throw new ObjectOptimisticLockingFailureException("some class", idOfRootRecord);
    }

    // // springの機能では、既に値が埋まっていると監査項目を更新してくれない・・・ので、監査項目を削除しておく。
    // // 厳密には、「一覧画面から単票画面への遷移」の場合は削除すると画面表示で使えなくなるので、
    // if (pattern == ActionPattern.UPDATE || pattern == ActionPattern.DELETE) {
    // putNullToAuditingFields(e);
    // }

    return e;
  }

  // public static enum ActionPattern {
  // NO_DATA_CHANGE, UPDATE, DELETE;
  // }
}
