package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.util.StringUtil;
import jp.ecuacion.splib.web.bean.RedirectUrlBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPageBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPathBean;
import jp.ecuacion.splib.web.bean.RequestResultBean;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibGeneralForm;
import jp.ecuacion.splib.web.service.SplibGeneralService;
import jp.ecuacion.splib.web.util.SplibSecurityUtil;
import jp.ecuacion.splib.web.util.SplibSecurityUtil.RolesAndAuthoritiesBean;
import jp.ecuacion.splib.web.util.SplibUtil;
import jp.ecuacion.splib.web.util.internal.TransactionTokenUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.lang.NonNull;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.servlet.resource.NoResourceFoundException;

public abstract class SplibGeneralController<G extends SplibGeneralForm>
    extends SplibBaseController {

  /** 通常constructorの引数とする項目だが、数が多いのでmethodChain方式で渡す形とした。 */
  public static class ControllerSettings {

    /** 敢えてpublicにはしない。newSettings()から取得する形とする。 */
    ControllerSettings() {

    }

    /**
     * その機能を表す名前。 form, recordなど各種クラスのprefix、urlの/xxx/searchなどの"xxx"部分などに使用。 複数の機能で同一の機能名を持たせるのは不可。
     * <p>
     * "accGroup"のようにentity名と同一にするのが推奨。（実装上機能名及び各クラスにentity名が冠されることになりわかりやすいため）
     * ただし、実際問題、同一entityを主として使用するが、複数画面用意する必要がある場合などに名前を変える必要があるため、
     * "accGroupManagement"などentity名とは異なる名前も指定可能。 主要となるEntityは同一だが用途が異なるため画面を分けたい、などの場合には本機能名を分ける。
     * 異なるControllerで同一のfunctionを指定することは不可。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
     * </p>
     */
    protected String function;

    /**
     * 機能にsearch-list-editの一連の機能がある場合に、その個々の機能（edit）に対する名称。
     * nullは不可、指定なしの場合は""を設定。同一機能で複数controllerが存在する場合以外は通常は指定なしで問題なし。
     */
    protected String subFunction = "";

    /**
     * 通常はhtmlFile名はfunction +
     * subFunctionで指定されるが、1画面に複数controllerを持つ場合は、個々のcontrollerのsubFunctionとhtmlファイル名の
     * postfixは異なる場合がある。 その場合に、postfixをhtmlFilePostfixで指定する。
     */
    protected String htmlFilenamePostfix;

    /**
     * form直下のrecordのfield名。対応するentity名を使用する。 特にlist-edit
     * templateでは、form配下には一つのみのrecordを保持するルールのため、entity名と同一にしておくのが推奨。 （list-edit
     * templateではそれ以外のパターンをや流必要性がなく未実施のためサポート外）
     * <p>
     * java側では、spring側ではrecordNameが必要だが毎回書くのが面倒な際に自動補完する目的などで使用。
     * また本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用。
     * </p>
     * <p>
     * page-generalの場合は、recordが存在しない場合もある。その場合は""を指定。（nullは不可）
     * </p>
     */
    protected String rootRecordName;

    public ControllerSettings function(String function) {
      this.function = function;
      return this;
    }

    public String function() {
      return function;
    }

    public ControllerSettings subFunction(String subFunction) {
      this.subFunction = subFunction;
      return this;
    }

    public String subFunction() {
      return subFunction;
    }

    public ControllerSettings htmlFilenamePostfix(String htmlFilenamePostfix) {
      this.htmlFilenamePostfix = htmlFilenamePostfix;
      return this;
    }

    public String htmlFilenamePostfix() {
      return htmlFilenamePostfix;
    }

    public ControllerSettings rootRecordName(String rootRecordName) {
      this.rootRecordName = rootRecordName;
      return this;
    }

    /** 未設定の場合はfunctionと同一となる。 */
    public String rootRecordName() {
      return rootRecordName == null ? function : rootRecordName;
    }
  }

  public static ControllerSettings newSettings() {
    return new ControllerSettings();
  }

  public static final String KEY = "controller";

  protected RedirectUrlBean redirectUrlOnAppExceptionBean;

  private ControllerSettings settings;

  @Autowired
  private SplibUtil util;

  private StringUtil strUtil = new StringUtil();

  public abstract SplibGeneralService getService();
  
  protected RolesAndAuthoritiesBean rolesAndAuthoritiesBean;

  /** functionを指定したconstructor。 */
  public SplibGeneralController(@Nonnull String function) {
    this(function, newSettings());
  }

  /** functionを指定したconstructor。functionだけは必須なのでconstructorの引数としている。 */
  protected SplibGeneralController(@Nonnull String function, @NonNull ControllerSettings settings) {
    settings.function(function);
    this.settings = settings;
  }

  // /** function, subFunctionを指定したconstructor。 */
  // public SplibGeneralController(@Nonnull String function, String subFunction) {
  // this(function, subFunction, null, function);
  // }
  //
  // /**
  // * function, htmlFilePostfix, recordNameを指定したconstructor。 htmlFilePostfixは、
  // */
  // public SplibGeneralController(@Nonnull String function, String htmlFilenamePostfix,
  // String rootRecordName) {
  // this(function, null, htmlFilenamePostfix, rootRecordName);
  //
  // }
  //
  // /**
  // * function, subFunction, htmlFilePostfix, recordNameを指定したconstructor。
  // * htmlFilePostfixは、通常はhtmlFile名はfunction + subFunctionで指定されるが、
  // * 1画面に複数controllerを持つ場合は、個々のcontrollerのsubFunctionとhtmlファイル名のpostfixは異なる場合がある。
  // * その場合に、postfixをhtmlFilePostfixで指定する。
  // */
  // public SplibGeneralController(@Nonnull String function, String subFunction,
  // String htmlFilenamePostfix, String rootRecordName) {
  // super();
  //
  // this.settings = new Settings().function(function).subFunction(subFunction)
  // .htmlFilenamePostfix(htmlFilenamePostfix).rootRecordName(rootRecordName);
  // }

  public String getFunction() {
    return settings.function();
  }

  public String getRootRecordName() {
    return settings.rootRecordName();
  }

  public RedirectUrlBean getRedirectUrlOnAppExceptionBean() {
    return redirectUrlOnAppExceptionBean;
  }

  /**
   * 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。
   * ちなみにmodel自体は、Controllerの保持する値に依存しないため、SplibControllerAdviceにてrequestに追加している。
   */
  @ModelAttribute
  private void setCommonParamsToModel(Model model, @AuthenticationPrincipal UserDetails loginUser) {
    model.addAttribute("function", settings.function());
    model.addAttribute("rootRecordName", settings.rootRecordName());

    rolesAndAuthoritiesBean =
        loginUser == null ? new SplibSecurityUtil().getRolesAndAuthoritiesBean()
            : new SplibSecurityUtil().getRolesAndAuthoritiesBean(loginUser);
    model.addAttribute("rolesAndAuthorities", rolesAndAuthoritiesBean);
  }

  /** メニューなどからURLを指定された際に表示する処理のreturnとして使用。htmlページのファイル名ルールを統一化する目的で使用。 */
  protected String getReturnStringToShowPage() {
    return getDefaultHtmlFileName();
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, getDefaultHtmlFileName(), true);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, boolean needsSuccessMessage) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, getDefaultHtmlFileName(), needsSuccessMessage);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, String page) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, page, true);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, String page, boolean needsSuccessMessage) {
    if (needsSuccessMessage) {
      // needsSuccessMesssage == trueの場合は、成功のメッセージが出るようmodelに追加
      RequestResultBean bean = (RequestResultBean) model.getAttribute(RequestResultBean.key);
      bean.setNeedsSuccessMessage(true);
    }

    return page;
  }

  /** 処理が終わり、最終的にredirectする場合に使用。 */
  protected String getReturnStringOnSuccess(RedirectUrlBean redirectUrlBean) {

    if (redirectUrlBean instanceof RedirectUrlPageBean) {
      return ((RedirectUrlPageBean) redirectUrlBean).getUrl(util.getLoginState(),
          settings.function(), getDefaultSubFunctionOnSuccess(), getDefaultPageOnSuccess());

    } else if (redirectUrlBean instanceof RedirectUrlPathBean) {
      return ((RedirectUrlPathBean) redirectUrlBean).getUrl();

    } else {
      throw new RuntimeException("RedirectUrlBeanが想定外の値です。" + redirectUrlBean);
    }
  }

  /**
   * validation errorなどが発生した際はredirectを通常しないので、ボタンを押した際の/.../action のパスがurlに残る。
   * その画面のactionはpostで受けているのに、そのurlをbrowserのurlバーを指定してenterしてしまう（つまりgetで送信してしまう）ことがままある。
   * システムエラーになるのはよくないので、その場合は404と同様の処理としてしまう。actionにparameterをつけないと全てここに来てしまうので注意。
   */
  @GetMapping(value = "action")
  public void throw404() throws NoResourceFoundException {
    throw new NoResourceFoundException(HttpMethod.GET, "from SplibGeneralController");
  }

  /** 場面によりget/post両方ありうるので両方記載しておく。 */
  @RequestMapping(value = "action", params = "submitOnChangeToRefresh=true",
      method = {RequestMethod.GET, RequestMethod.POST})
  public String submitOnChangeToRefresh(G form, Model model) throws Exception {
    prepare(model, form);
    model.addAttribute(getFunction() + strUtil.capitalize(settings.subFunction()) + "Form", form);
    getService().prepareForm(form);
    return settings.function() + strUtil.capitalize(settings.subFunction());
  }

  /** 本controllerとペアになる画面htmlの文字列。基本は&lt;function&gt;.html。 */
  public String getDefaultHtmlFileName() {
    return settings.function()
        + strUtil.capitalize(settings.htmlFilenamePostfix() == null ? settings.subFunction()
            : settings.htmlFilenamePostfix());
  }

  /** 処理成功時redirectをする場合のredirect先subFunctionのdefault。 */
  public String getDefaultSubFunctionOnSuccess() {
    return settings.subFunction();
  }

  /** 処理成功時redirectをする場合のredirect先subFunctionのdefault。 */
  public String getDefaultSubFunctionOnAppException() {
    return settings.subFunction();
  }

  /** 処理成功時redirectをする場合のredirect先pageのdefault。 */
  public String getDefaultPageOnSuccess() {
    return "page";
  }

  /** 処理成功時redirectをする場合のredirect先pageのdefault。 */
  public String getDefaultPageOnAppException() {
    return "page";
  }

  /** validationチェックなし、redirectの場合は以下を使用。 */
  protected void prepare() throws InputValidationException, AppException {
    prepare(null, null, null, false);
  }

  /**
   * validationチェックなしの場合は以下を使用。
   */
  protected void prepare(Model model, SplibGeneralForm... nonValidationTargetForms)
      throws InputValidationException, AppException {
    prepare(model, null, null, false, nonValidationTargetForms);
  }

  /**
   * validationチェックありの場合は以下を使用。
   */
  protected void prepare(Model model, SplibGeneralForm validationTargetForm, BindingResult result,
      SplibGeneralForm... nonValidationTargetForms) throws InputValidationException, AppException {
    prepare(model, validationTargetForm, result, true, nonValidationTargetForms);
  }

  /**
   * エラー処理などに必要な処理を行う。 本処理は、@XxxMappingにより呼び出されるメソッド全てで呼び出す必要あり。
   * validation・BLチェック含めエラー発生なし、かつredirect、かつtransactionTokenCheck不要、の場合は厳密にはチェックは不要となる。
   * が、最低でも引数なしのメソッドは呼ぶ（=transactionTokenCheckは実施）ルールとし、transactionTokenCheckが不要の場合は別途それを設定することとする
   */
  private void prepare(Model model, SplibGeneralForm validationTargetForm, BindingResult result,
      boolean needsValidationCheck, SplibGeneralForm... nonValidationTargetForms)
      throws InputValidationException, AppException {

    // nonValidationTargetFormsは、"..."の場合nullにはならず0個の配列になっていたのでnullチェックは不要。
    List<SplibGeneralForm> allFormList = new ArrayList<>(Arrays.asList(nonValidationTargetForms));
    if (validationTargetForm != null) {
      allFormList.add(validationTargetForm);
    }

    // 全form共通処理
    for (SplibGeneralForm form : allFormList) {
      // formをmodelに追加
      model.addAttribute(form);

      // submitされない、毎回再設定が必要な値（selectの選択肢一覧など）を設定
      getService().prepareForm(form);
    }

    // エラー処理のためにmodelにcontrollerを格納
    request.setAttribute(KEY, this);

    transactionTokenCheck();

    if (needsValidationCheck) {
      validationCheck(validationTargetForm, result, util.getLoginState(), rolesAndAuthoritiesBean);
    }
  }

  /**
   * 以下を実施。 設定項目の値に従ってpulldownの選択肢を動的に変更したい場合など、敢えてvalidation checkをしたくない場面もあるので、 validation
   * checkの要否を持たせている。
   * <ul>
   * <li>transactionToken check</li>
   * <li>validation check</li>
   * </ul>
   */
  protected void transactionTokenCheck() throws CustomizedValidationAppException {
    // transactionToken check
    String tokenFromHtml =
        (String) request.getParameter(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    @SuppressWarnings("unchecked")
    Set<String> tokenSet = (Set<String>) request.getSession()
        .getAttribute(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    if (tokenSet != null && tokenFromHtml != null) {
      if (!tokenSet.contains(tokenFromHtml)) {

        String msgId = "jp.ecuacion.splib.web.common.message.tokenInvalidate";
        throw new CustomizedValidationAppException(msgId);
      }

      tokenSet.remove(tokenFromHtml);
    }
  }

  private void validationCheck(SplibGeneralForm form, BindingResult result, String loginState,
      RolesAndAuthoritiesBean bean) throws InputValidationException {
    // input validation
    boolean hasNotEmptyError = false;
    hasNotEmptyError = form.hasNotEmptyError(loginState, bean);

    if (hasNotEmptyError || (result != null && result.hasErrors())) {
      throw new InputValidationException(form);
    }
  }
}
