package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.splib.core.form.record.SplibRecord;
import jp.ecuacion.splib.web.bean.RedirectUrlBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPageOnSuccessBean;
import jp.ecuacion.splib.web.constant.SplibWebConstants;
import jp.ecuacion.splib.web.form.SplibListForm;
import jp.ecuacion.splib.web.form.SplibSearchForm;
import jp.ecuacion.splib.web.service.SplibSearchListService;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;

public abstract class SplibSearchListController
    <S extends SplibSearchForm, L extends SplibListForm<? extends SplibRecord>>
    extends SplibGeneralController {

  protected abstract S getNewSearchForm();

  public abstract SplibSearchListService getService();

  public SplibSearchListController(@Nonnull String function) {
    super(function, "searchList", function);
  }

  public SplibSearchListController(@Nonnull String function, String rootRecordName) {
    super(function, "searchList", rootRecordName);
  }

  @Override
  public String getDefaultSubFunctionOnSuccess() {
    return "searchList";
  }

  @GetMapping(value = "page")
  public String page(Model model, S f) throws Exception {
    S searchForm = getProperSearchForm(model, f);

    SplibListForm<? extends SplibRecord> listForm = getService().getListForm(searchForm);
    listForm.setDataKind(searchForm.getDataKind());
    model.addAttribute(function + "ListForm", listForm);

    return getReturnStringOnShowingPage();
  }

  // /** list画面での削除時のエラー発生など、エラーがあって一覧画面に戻る場合に使用。 */
  // public String search(Model model, String dataKind) throws Exception {
  // S form = getNewSearchForm();
  // form.setDataKind(dataKind);
  //
  // return getReturnStringOnShowingPage();
  // }

  @GetMapping(value = "action", params = "search")
  public String search(Model model, S f) throws Exception {
    return page(model, f);
  }

  @GetMapping(value = "action", params = "action=searchAgain")
  public String searchAgain(Model model, S f) throws Exception {
    return search(model, f);
  }


  @SuppressWarnings({"unchecked"})
  protected S getProperSearchForm(Model model, S f) {

    String formName = function + "SearchForm";
    String key = getSessionKey(formName, f);
    if (f != null) {
      // 検索画面上で検索された場合はその条件を採用し保管。
      // それ以外の場合（メニューのリンクなどから来た場合など）はmodelに設定されたものをそのまま使用。（ここでは何もしない）
      if (f.isRequestFromSearchForm()) {
        request.getSession().setAttribute(key, f);
      }

      // 初回アクセスでsessionに存在しない場合は引数のformを使用
      if (request.getSession().getAttribute(key) == null) {
        request.getSession().setAttribute(key, f);
      }

    } else {
      // その前に検索しているはずなので、引数のfがnullで、かつsessionにも情報がない、という場合はあり得ない。
      if (request.getSession().getAttribute(formName) == null) {
        throw new RuntimeException("f == null cannot be occurred.");
      }
    }

    S formUsedForSearch = (S) request.getSession().getAttribute(getSessionKey(formName, f));

    // 使用するsearchFormがspring mvcからも使用されるよう、modelにも入れておく
    model.addAttribute(formName, formUsedForSearch);

    return formUsedForSearch;
  }

  /** fがnullの場合null Pointerが発生するためメソッド冒頭で定義することはできず、別メソッドとした。 */
  private String getSessionKey(String formName, S f) {
    return formName + (f == null || f.getDataKind() == null || f.getDataKind().equals("") ? ""
        : "." + f.getDataKind());
  }

  @GetMapping(value = "action", params = "conditionClear")
  public String searchConditionClear(Model model, S f) throws MultipleAppException {
    // 情報をクリアするための設定を行う
    String formName = function + "SearchForm";
    String sessionKey = formName
        + (f.getDataKind() == null || f.getDataKind().equals("") ? "" : "." + f.getDataKind());
    request.getSession().setAttribute(sessionKey, null);

    return getReturnStringOnSuccess(new RedirectUrlPageOnSuccessBean().noSuccessMessage()
        .putParam(SplibWebConstants.KEY_DATA_KIND, f.getDataKind()));
  }

  /**
   * 一覧からの削除処理。 delete用URLをお気に入りに入れられても困るので、本当はPostでの通信にしたいところだったが、
   * 呼び出し先を、showUpdateForm()と同じ「action」にしており、その状態でdeleteの場合のみpostに変更して投げてもspring mvcが判別つかないようで、
   * 「postは無効です」的なエラーが発生する。
   * deleteのみ別のformとするのもやりにくいし、formで持つactionをjavascriptで書き換えるのも微妙（th:actionでthymeleafが制御しているところなので）
   * なことから、GETとすることを許容する。以下の対策を行うことで、実質問題も起こらないと考えられる。 -
   * PRGを使うことで、delete中にシステムエラーでも起きない限り、delete時のURLがブラウザのURLバーには残らない
   */
  @GetMapping(value = "action", params = "delete")
  public String delete(Model model, L f, @AuthenticationPrincipal UserDetails loginUser)
      throws Exception {
    prepare(model, f);
    getService().delete(f, loginUser);

    return getReturnStringOnSuccess(new RedirectUrlPageOnSuccessBean()
        .putParam(SplibWebConstants.KEY_DATA_KIND, f.getDataKind()));
  }

  @GetMapping(value = "action", params = "showInsertForm")
  public String showInsertForm(Model model) {
    RedirectUrlBean bean = new RedirectUrlPageOnSuccessBean("edit", "page").noSuccessMessage()
        .putParamMap(request.getParameterMap());
    return getReturnStringOnSuccess(bean);
  }

  @GetMapping(value = "action", params = "showUpdateForm")
  public String showUpdateForm(Model model) {
    RedirectUrlBean bean = new RedirectUrlPageOnSuccessBean("edit", "page").noSuccessMessage()
        .putParamMap(request.getParameterMap());
    return getReturnStringOnSuccess(bean);
  }
}
