package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import java.util.Set;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.util.StringUtil;
import jp.ecuacion.splib.web.advice.SplibExceptionHandler;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerData;
import jp.ecuacion.splib.web.bean.RedirectUrlBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPageBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPathBean;
import jp.ecuacion.splib.web.bean.RequestResultBean;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibGeneralForm;
import jp.ecuacion.splib.web.service.SplibGeneralService;
import jp.ecuacion.splib.web.util.SplibUtil;
import jp.ecuacion.splib.web.util.internal.TransactionTokenUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibGeneralController extends SplibBaseController {

  /**
   * その機能を表す名前。 form, recordなど各種クラスのprefix、urlの/xxx/searchなどの"xxx"部分などに使用。 複数の機能で同一の機能名を持たせるのは不可。
   * <p>
   * "accGroup"のようにentity名と同一にするのが推奨。（実装上機能名及び各クラスにentity名が冠されることになりわかりやすいため）
   * ただし、実際問題、同一entityを主として使用するが、複数画面用意する必要がある場合などに名前を変える必要があるため、
   * "accGroupManagement"などentity名とは異なる名前も指定可能。 主要となるEntityは同一だが用途が異なるため画面を分けたい、などの場合には本機能名を分ける。
   * 異なるControllerで同一のfunctionを指定することは不可。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
   * </p>
   */
  protected String function;

  /**
   * 機能にsearch-list-editの一連の機能がある場合に、その個々の機能（edit）に対する名称。
   * nullは不可、指定なしの場合は""を設定。同一機能で複数controllerが存在する場合以外は通常は指定なしで問題なし。
   */
  protected String subFunction = "";

  /**
   * 
   */
  protected String htmlFilenamePostfix;

  /**
   * form直下のrecordのfield名。対応するentity名を使用する。 特にlist-edit
   * templateでは、form配下には一つのみのrecordを保持するルールのため、entity名と同一にしておくのが推奨。 （list-edit
   * templateではそれ以外のパターンをや流必要性がなく未実施のためサポート外）
   * <p>
   * java側では、spring側ではrecordNameが必要だが毎回書くのが面倒な際に自動補完する目的などで使用。
   * また本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用。
   * </p>
   * <p>
   * page-generalの場合は、recordが存在しない場合もある。その場合は""を指定。（nullは不可）
   * </p>
   */
  @Nonnull
  protected String rootRecordName;

  protected RedirectUrlBean redirectUrlOnAppExceptionBean;

  @Autowired
  private SplibUtil util;

  private StringUtil strUtil = new StringUtil();

  public abstract SplibGeneralService getService();

  /** functionを指定したconstructor。 */
  public SplibGeneralController(@Nonnull String function) {
    this(function, null);
  }

  /** function, functionPosxfixを指定したconstructor。 */
  public SplibGeneralController(@Nonnull String function, String subFunction) {
    this(function, subFunction, null, function);
  }

  /**
   * function, htmlFilePostfix, recordNameを指定したconstructor。 htmlFilePostfixは、通常はhtmlFile名はfunction +
   * functionPosxfixで指定されるが、1画面に複数controllerを持つ場合は、個々のcontrollerのsubFunctionとhtmlファイル名の
   * postfixは異なる場合がある。 その場合に、postfixをhtmlFilePostfixで指定する。
   */
  public SplibGeneralController(@Nonnull String function, String htmlFilenamePostfix,
      String recordName) {
    super();
    this.function = function;
    this.htmlFilenamePostfix = htmlFilenamePostfix;
    this.rootRecordName = recordName == null ? function : recordName;
  }

  /**
   * function, functionPosxfix、htmlFilePostfix, recordNameを指定したconstructor。
   * htmlFilePostfixは、通常はhtmlFile名はfunction + functionPosxfixで指定されるが、
   * 1画面に複数controllerを持つ場合は、個々のcontrollerのsubFunctionとhtmlファイル名のpostfixは異なる場合がある。
   * その場合に、postfixをhtmlFilePostfixで指定する。
   */
  public SplibGeneralController(@Nonnull String function, String subFunction,
      String htmlFilenamePostfix, String recordName) {
    super();
    this.function = function;
    this.subFunction = subFunction == null ? "" : subFunction;
    this.htmlFilenamePostfix = htmlFilenamePostfix;
    this.rootRecordName = recordName == null ? function : recordName;
  }

  public String getFunction() {
    return function;
  }

  public String getRootRecordName() {
    return rootRecordName;
  }

  public RedirectUrlBean getRedirectUrlOnAppExceptionBean() {
    return redirectUrlOnAppExceptionBean;
  }

  /**
   * 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。
   * ちなみにmodel自体は、Controllerの保持する値に依存しないため、SplibControllerAdviceにてrequestに追加している。
   */
  @ModelAttribute
  private void setCommonParamsToModel(Model model) {
    model.addAttribute("function", function);
    model.addAttribute("rootRecordName", rootRecordName);
  }

  /** メニューなどからURLを指定された際に表示する処理のreturnとして使用。htmlページのファイル名ルールを統一化する目的で使用。 */
  protected String getReturnStringOnShowingPage() {
    return getDefaultHtmlFileName();
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, getDefaultHtmlFileName(), true);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, boolean needsSuccessMessage) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, getDefaultHtmlFileName(), needsSuccessMessage);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, String page) {
    // defaultではsuccess messageを表示
    return getReturnStringOnSuccess(model, page, true);
  }

  /** 処理が終わり、最終的にpageを表示する（redirectしない）場合に使用。 */
  protected String getReturnStringOnSuccess(Model model, String page, boolean needsSuccessMessage) {
    if (needsSuccessMessage) {
      // needsSuccessMesssage == trueの場合は、成功のメッセージが出るようmodelに追加
      RequestResultBean bean = (RequestResultBean) model.getAttribute(RequestResultBean.key);
      bean.setNeedsSuccessMessage(true);
    }

    return page;
  }

  /** 処理が終わり、最終的にredirectする場合に使用。 */
  protected String getReturnStringOnSuccess(RedirectUrlBean redirectUrlBean) {

    if (redirectUrlBean instanceof RedirectUrlPageBean) {
      return ((RedirectUrlPageBean) redirectUrlBean).getUrl(util.getLoginState(), function,
          getDefaultSubFunctionOnSuccess(), getDefaultPageOnSuccess());

    } else if (redirectUrlBean instanceof RedirectUrlPathBean) {
      return ((RedirectUrlPathBean) redirectUrlBean).getUrl();

    } else {
      throw new RuntimeException("RedirectUrlBeanが想定外の値です。" + redirectUrlBean);
    }
  }

  /** 場面によりget/post両方ありうるので両方記載しておく。 */
  @GetMapping(value = "action", params = "submitOnChangeToRefresh=true")
  @PostMapping(value = "action", params = "submitOnChangeToRefresh=true")
  public String submitOnChangeToRefresh(SplibGeneralForm f, Model model) throws Exception {
    prepare(f, model);
    model.addAttribute(function + strUtil.capitalize(subFunction) + "Form", f);
    getService().prepareForm(f);
    return function + strUtil.capitalize(subFunction);
  }

  /** 本controllerとペアになる画面htmlの文字列。基本は&lt;function&gt;.html。 */
  public String getDefaultHtmlFileName() {
    return function
        + strUtil.capitalize(htmlFilenamePostfix == null ? subFunction : htmlFilenamePostfix);
  }

  /** 処理成功時redirectをする場合のredirect先subFunctionのdefault。 */
  public String getDefaultSubFunctionOnSuccess() {
    return subFunction;
  }

  /** 処理成功時redirectをする場合のredirect先subFunctionのdefault。 */
  public String getDefaultSubFunctionOnAppException() {
    return subFunction;
  }

  /** 処理成功時redirectをする場合のredirect先pageのdefault。 */
  public String getDefaultPageOnSuccess() {
    return "page";
  }

  /** 処理成功時redirectをする場合のredirect先pageのdefault。 */
  public String getDefaultPageOnAppException() {
    return "page";
  }

  /**
   * 以下を実施。 設定項目の値に従ってpulldownの選択肢を動的に変更したい場合など、敢えてvalidation checkをしたくない場面もあるので、 validation
   * checkの要否を持たせている。
   * <ul>
   * <li>transactionToken check</li>
   * <li>validation check</li>
   * </ul>
   */
  protected void commonProc(SplibGeneralForm form, Model model, BindingResult result,
      boolean needsValidationCheck) throws InputValidationException, AppException {

    tokenCheck();

    if (needsValidationCheck) {
      validationCheck(form, result);
    }
  }

  protected void tokenCheck() throws CustomizedValidationAppException {
    // transactionToken check
    String tokenFromHtml =
        (String) request.getParameter(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    @SuppressWarnings("unchecked")
    Set<String> tokenSet = (Set<String>) request.getSession()
        .getAttribute(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    if (tokenSet != null && tokenFromHtml != null) {
      if (!tokenSet.contains(tokenFromHtml)) {

        String msgId = "jp.ecuacion.splib.web.common.message.tokenInvalidate";
        throw new CustomizedValidationAppException(msgId);
      }

      tokenSet.remove(tokenFromHtml);
    }
  }

  private void validationCheck(SplibGeneralForm form, BindingResult result)
      throws InputValidationException {
    // input validation
    boolean hasNotEmptyError = false;
    hasNotEmptyError = form.hasNotEmptyError();

    if (hasNotEmptyError || (result != null && result.hasErrors())) {
      throw new InputValidationException(form);
    }
  }

  /**
   * validationチェックなし、エラー終了後の画面遷移なし、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model)
      throws InputValidationException, AppException {
    prepare(inputForm, model, null, false);
  }

  /**
   * validationチェックあり、エラー終了後の画面遷移なし、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model, BindingResult result)
      throws InputValidationException, AppException {
    prepare(inputForm, model, result, true);
  }

  /**
   * 本メソッドは以下の処理を行う。 input validationは個別処理で書いても良いのだが、それも含めて1行でかけるものを用意した。
   * 
   * <p>
   * 1. AppExceptionに対するメッセージ出力をControllerの各処理で行うのではなく@ControllerAdviceで行うための準備
   * - @ControllerAdvice側でmodelをDIしても中身が空のため、controller側のものを保管しておく <br>
   * - エラー発生時の戻り先ページを設定しておく
   * </p>
   * <p>
   * commonProcの実行。
   * </p>
   */
  private void prepare(SplibGeneralForm inputForm, Model model,
      BindingResult result, boolean needsValidationCheck)
      throws InputValidationException, AppException {

    // submitされない値（selectの選択肢一覧など）を取得する
    getService().prepareForm(inputForm);

    // 別画面に遷移した場合は使わない場合もあるが、同一画面に遷移する場合の同一formは自動で設定しておく
    model.addAttribute(function + strUtil.capitalize(subFunction) + "Form", inputForm);

    // エラー処理用の準備
    SplibExceptionHandlerData info =
        new SplibExceptionHandlerData(this, inputForm);
    request.setAttribute(SplibExceptionHandler.INFO_FOR_ERROR_HANDLING, info);

    commonProc(inputForm, model, result, needsValidationCheck);
  }
}
