package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import java.util.Set;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.splib.web.advice.SplibExceptionHandler;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerData;
import jp.ecuacion.splib.web.bean.RedirectUrlOnAppExceptionBean;
import jp.ecuacion.splib.web.bean.RedirectUrlOnSuccessBean;
import jp.ecuacion.splib.web.bean.RedirectUrlOnSuccessPageBean;
import jp.ecuacion.splib.web.bean.RedirectUrlOnSuccessPathBean;
import jp.ecuacion.splib.web.enums.LoginStateEnum;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibGeneralForm;
import jp.ecuacion.splib.web.service.SplibGeneralService;
import jp.ecuacion.splib.web.util.TransactionTokenUtil;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibGeneralController extends SplibBaseController {

  /**
   * その機能を表す名前。 form, recordなど各種クラスのprefix、urlの/xxx/searchなどの"xxx"部分などに使用。 複数の機能で同一の機能名を持たせるのは不可。
   * <p>
   * "accGroup"のようにentity名と同一にするのが推奨。（実装上機能名及び各クラスにentity名が冠されることになりわかりやすいため）
   * ただし、実際問題、同一entityを主として使用するが、複数画面用意する必要がある場合などに名前を変える必要があるため、
   * "accGroupManagement"などentity名とは異なる名前も指定可能。 主要となるEntityは同一だが用途が異なるため画面を分けたい、などの場合には本機能名を分ける。
   * 異なるControllerで同一のfunctionNameを指定することは不可。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
   * </p>
   */
  protected String functionName;

  /**
   * form直下のrecordのfield名。対応するentity名を使用する。 特にlist-edit
   * templateでは、form配下には一つのみのrecordを保持するルールのため、entity名と同一にしておくのが推奨。 （list-edit
   * templateではそれ以外のパターンをや流必要性がなく未実施のためサポート外）
   * <p>
   * java側では、spring側ではrecordNameが必要だが毎回書くのが面倒な際に自動補完する目的などで使用。
   * また本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用。
   * </p>
   * <p>
   * page-generalの場合は、recordが存在しない場合もある。その場合は""を指定。（nullは不可）
   * </p>
   */
  @Nonnull
  protected String rootRecordName;

  protected RedirectUrlOnAppExceptionBean redirectUrlOnAppExceptionBean;

  public abstract SplibGeneralService getService();

  /** functionNameを指定。 */
  public SplibGeneralController(LoginStateEnum loginState, @Nonnull String functionName) {
    super(loginState);
    this.functionName = functionName;
    this.rootRecordName = functionName;
  }

  /** functionName, rootRecordNameを指定。 */
  public SplibGeneralController(LoginStateEnum loginState, @Nonnull String functionName,
      String rootRecordName) {
    this(loginState, functionName);
    this.rootRecordName = rootRecordName == null ? "" : rootRecordName;
  }

  public String getFunctionName() {
    return functionName;
  }

  public String getRootRecordName() {
    return rootRecordName;
  }

  public RedirectUrlOnAppExceptionBean getRedirectUrlOnAppExceptionBean() {
    return redirectUrlOnAppExceptionBean;
  }

  /**
   * 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。
   * ちなみにmodel自体は、Controllerの保持する値に依存しないため、SplibControllerAdviceにてrequestに追加している。
   */
  @ModelAttribute
  private void setCommonParamsToModel(Model model) {
    model.addAttribute("functionName", functionName);
    model.addAttribute("rootRecordName", rootRecordName);
  }

  /** 処理が終わり、最終的にredirectする場合に使用。 */
  protected String getActionSuccessRedirectString(RedirectUrlOnSuccessBean redirectUrlBean) {

    if (redirectUrlBean instanceof RedirectUrlOnSuccessPageBean) {
      return ((RedirectUrlOnSuccessPageBean) redirectUrlBean).getUrl(loginState, functionName);

    } else if (redirectUrlBean instanceof RedirectUrlOnSuccessPathBean) {
      return ((RedirectUrlOnSuccessPathBean) redirectUrlBean).getUrl();

    } else {
      throw new RuntimeException("RedirectUrlBeanが想定外の値です。" + redirectUrlBean);
    }
  }

  @PostMapping(value = "action", params = "submitOnChangeToRefresh=true")
  public String updateOptions(SplibGeneralForm f, Model model) throws Exception {
    prepare(f, model);
    model.addAttribute(functionName + "EditForm", f);
    model.addAttribute("testkey", "testvalue");
    model.addAttribute(functionName + "EditForm", f);
    model.addAttribute(functionName + "Edit2Form", f);
    getService().prepareForm(f);
    return functionName + "Edit";
  }

  /** 本controllerとペアになる画面htmlの文字列。基本は&lt;functionName&gt;.html。 */
  public String getDefaultHtmlFileName() {
    return functionName;
  }

  /** 処理成功時の表示画面のdefault。 */
  public String getDefaultSuccessPage() {
    return "page";
  }

  /** エラー時の表示画面のdefault。 */
  public String getDefaultErrorPage() {
    return "page";
  }

  /**
   * 以下を実施。 設定項目の値に従ってpulldownの選択肢を動的に変更したい場合など、敢えてvalidation checkをしたくない場面もあるので、 validation
   * checkの要否を持たせている。
   * <ul>
   * <li>transactionToken check</li>
   * <li>validation check</li>
   * </ul>
   */
  protected void commonProc(SplibGeneralForm form, Model model, BindingResult result,
      boolean needsValidationCheck) throws InputValidationException, AppException {

    tokenCheck();

    if (needsValidationCheck) {
      validationCheck(form, result);
    }
  }

  protected void tokenCheck() throws CustomizedValidationAppException {
    // transactionToken check
    String tokenFromHtml =
        (String) request.getParameter(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    @SuppressWarnings("unchecked")
    Set<String> tokenSet = (Set<String>) request.getSession()
        .getAttribute(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    if (tokenSet != null && tokenFromHtml != null) {
      if (!tokenSet.contains(tokenFromHtml)) {

        String msgId = "jp.ecuacion.splib.web.common.message.tokenInvalidate";
        throw new CustomizedValidationAppException(msgId);
      }

      tokenSet.remove(tokenFromHtml);
    }
  }

  private void validationCheck(SplibGeneralForm form, BindingResult result)
      throws InputValidationException {
    // input validation
    boolean hasNotEmptyError = false;
    hasNotEmptyError = form.hasNotEmptyError();

    if (hasNotEmptyError || (result != null && result.hasErrors())) {
      throw new InputValidationException(form);
    }
  }

  /**
   * validationチェックなし、エラー終了後の画面遷移なし、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model)
      throws InputValidationException, AppException {
    prepare(inputForm, model, null, null, false);
  }

  /**
   * validationチェックなし、エラー終了後の画面遷移あり、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model, String nextPageOnError)
      throws InputValidationException, AppException {
    prepare(inputForm, model, nextPageOnError, null, false);
  }

  /**
   * validationチェックあり、エラー終了後の画面遷移なし、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model, BindingResult result)
      throws InputValidationException, AppException {
    prepare(inputForm, model, null, result, true);
  }

  /**
   * validationチェックあり、エラー終了後の画面遷移あり、の場合は以下を使用。
   */
  protected void prepare(SplibGeneralForm inputForm, Model model, String nextPageOnError,
      BindingResult result) throws InputValidationException, AppException {
    prepare(inputForm, model, nextPageOnError, result, true);
  }

  /**
   * 本メソッドは以下の処理を行う。 input validationは個別処理で書いても良いのだが、それも含めて1行でかけるものを用意した。
   * 
   * <p>
   * 1. AppExceptionに対するメッセージ出力をControllerの各処理で行うのではなく@ControllerAdviceで行うための準備
   * - @ControllerAdvice側でmodelをDIしても中身が空のため、controller側のものを保管しておく <br>
   * - エラー発生時の戻り先ページを設定しておく
   * </p>
   * <p>
   * commonProcの実行。
   * </p>
   */
  private void prepare(SplibGeneralForm inputForm, Model model, String nextPageOnError,
      BindingResult result, boolean needsValidationCheck)
      throws InputValidationException, AppException {

    // 別画面に遷移した場合は使わない場合もあるが、同一画面に遷移する場合の同一formは自動で設定しておく
    model.addAttribute(functionName + "Form", inputForm);

    // エラー処理用の準備
    SplibExceptionHandlerData info =
        new SplibExceptionHandlerData(this, inputForm, nextPageOnError);
    request.setAttribute(SplibExceptionHandler.INFO_FOR_ERROR_HANDLING, info);

    commonProc(inputForm, model, result, needsValidationCheck);
  }
}
