package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.splib.web.advice.SplibExceptionHandler;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerData;
import jp.ecuacion.splib.web.enums.LoginStateEnum;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibEditForm;
import jp.ecuacion.splib.web.form.SplibForm;
import jp.ecuacion.splib.web.service.SplibEditService;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibEditController<E extends SplibEditForm>
    extends SplibGeneralController<E> {

  public abstract SplibEditService getService();

  public SplibEditController(LoginStateEnum loginState, @Nonnull String functionName) {
    super(loginState, functionName);
  }

  public SplibEditController(LoginStateEnum loginState, @Nonnull String functionName,
      String rootRecordName) {
    super(loginState, functionName, rootRecordName);
  }

  protected String getFunctionNamePostfix() {
    return "edit";
  }

  protected String getActionSuccessRedirectString() {
    return "redirect:"
        + (getEditSuccessRedirectPath() == null
            ? "/" + loginState.getCode() + "/" + functionName + "/" + getDefaultSuccessPage()
                + "?success"
            : getEditSuccessRedirectPath());
  }

  /** 処理成功時の表示画面のdefault。 */
  public String getDefaultSuccessPage() {
    return "search";
  }

  @GetMapping(value = "edit", params = "showInsertForm")
  public String showInsertForm(Model model, E f) throws AppException {
    prepareWithMovingToListOnError(f, model);

    SplibEditForm form = getService().getInsertForm(f);
    form.setIsInsert(true);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  @GetMapping(value = "edit", params = "showUpdateForm")
  public String showUpdateForm(Model model, E f) throws Exception {
    prepareWithMovingToListOnError(f, model);

    SplibEditForm form = getService().getUpdateForm(f);
    form.setIsInsert(false);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  protected String getUrlParams(String menuName, boolean needsAmpersand) {
    String additionalString = "";
    if (menuName != null && !menuName.equals("")) {
      additionalString = (needsAmpersand ? "&" : "") + "menuName=" + menuName;
    }

    return additionalString;
  }

  @PostMapping(value = "edit")
  public String edit(@Validated E f, BindingResult result, Model model,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(f, model, functionName + "Edit", result);
    getService().edit(f, loginUser);

    return getActionSuccessRedirectString() + getUrlParams(f.getMenuName(), true);
  }

  @PostMapping(value = "edit", params = "back")
  public String back(@Validated E f, BindingResult result, Model model) {
    return "redirect:/" + loginState.getCode() + "/" + functionName + "/search?"
        + getUrlParams(f.getMenuName(), false);
  }

  @PostMapping(value = "edit", params = "updateDropDown=true")
  public String updateOptions(E f, Model model) throws Exception {
    prepare(f, model, functionName + "Edit");
    model.addAttribute(functionName + "EditForm", f);
    return functionName + "Edit";
  }

  @Override
  protected void prepare(SplibForm form, Model model, String nextPageOnError, BindingResult result)
      throws InputValidationException, AppException {

    // resultがnullでない場合はもちろんそうだが、その他チェックでAppExceptionが発生した場合も、再表示のために情報が必要となるので基本呼び出しておく
    // （本当は、エラーが発生した場合にのみexceptionHandlerで呼び出すのがより好ましいのだが若干煩雑になるので一旦は毎回実施）
    if (form instanceof SplibEditForm) {
      // editの際にsubmitではこない値（selectの選択肢一覧など）を再度設定する
      getService().prepareEditForm((SplibEditForm) form);
    }

    super.prepare(form, model, nextPageOnError, result);
  }

  /**
   * 一覧で対象を指定し削除、など、list画面での処理の場合、formには削除対象のデータが入っているだけで一覧全体の情報は持っていない。
   * そのため画面を再表示するには、エラーメッセージを詰めたmodelを引き渡しつつ、list表示用の処理を呼び出し再度一覧情報を取得する流れとなる。
   * <p>
   * その場合、削除などの処理終了後にlist表示用の処理を呼び出すことになるのだが、reflectionを使用しそのメソッドを引数に渡してもらう、
   * というのも煩雑なので、一覧表示用のlistは「public String list(Model model)」と固定し、そこに戻る処理とする。
   * </p>
   */
  protected void prepareWithMovingToListOnError(SplibForm form, Model model, BindingResult result)
      throws InputValidationException, AppException {
    // 1. preparation
    SplibExceptionHandlerData info = new SplibExceptionHandlerData(this, form);
    request.setAttribute(SplibExceptionHandler.INFO_FOR_ERROR_HANDLING, info);

    commonProc(form, model, result, false);
  }

  /**
   * formチェックがない場合は、input validationのないこちらを使用。
   */
  protected void prepareWithMovingToListOnError(SplibForm form, Model model)
      throws InputValidationException, AppException {
    prepareWithMovingToListOnError(form, model, null);
  }
}
