package jp.ecuacion.splib.web.service;

import static jp.ecuacion.splib.web.service.SplibListJpaService.ActionPattern.DELETE;
import static jp.ecuacion.splib.web.service.SplibListJpaService.ActionPattern.EDIT;
import jakarta.servlet.http.HttpServletRequest;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.lib.jpa.dbaccess.EntityManagerWrapper;
import jp.ecuacion.splib.core.container.DatetimeFormatParameters;
import jp.ecuacion.splib.core.form.record.SplibRecord;
import jp.ecuacion.splib.jpa.repository.SplibRepository;
import jp.ecuacion.splib.web.form.SplibEditForm;
import jp.ecuacion.splib.web.form.SplibListForm;
import jp.ecuacion.splib.web.form.SplibSearchConditionForm;
import jp.ecuacion.splib.web.util.SplibJpaServiceUtil;
import jp.ecuacion.splib.web.util.SplibUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.orm.ObjectOptimisticLockingFailureException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.transaction.annotation.Transactional;

@Transactional(rollbackFor = Exception.class)
public abstract class SplibListJpaService<E extends AbstractEntity> extends SplibListService {

  @Autowired
  private HttpServletRequest request;

  @Autowired
  private SplibJpaServiceUtil<E> jpaServiceUtil;

  protected EntityManagerWrapper em;

  //
  // common
  //

  /**
   * offsetはlogin画面でのonload時に呼ばれるため、login画面を開いた状態で放置した場合は値がnullでエラーになる。
   */
  public DatetimeFormatParameters getParams() {
    return new SplibUtil().getParams(request);
  }

  /**
   * record内のlocalDate項目（String）をLocalDate形式で取得。
   */
  protected LocalDate localDate(String date) {
    return (date == null || date.equals("")) ? null
        : LocalDate.parse(date, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
  }

  public void setEntityManager(EntityManagerWrapper em) {
    this.em = em;
  }

  //
  // common: optimistic locking関連
  //

  /** findAndOptimisticLockingCheck() にて使用。selectのためのrepositoryを取得。 */
  protected abstract SplibRepository<E, Long> getRepositoryForOptimisticLocking();

  /** findAndOptimisticLockingCheck() にて使用。楽観的排他制御のためのversionを取得。 */
  protected abstract Object getVersion(E e);

  /**
   * edit画面への遷移・編集時・削除時のそれぞれで、DBからentityを取得する処理。 （edit画面への遷移では変更画面で画面表示するため、編集時・削除時はDB更新するため）
   * 取得と同時に楽観的排他制御エラー・同時にどうレコード削除時の制
   */
  protected AbstractEntity findAndOptimisticLockingCheck(String id, String versionInScreen,
      ActionPattern pattern) throws AppException {
    Optional<E> optional = getRepositoryForOptimisticLocking().findById(Long.valueOf(id));

    // 削除機能には、削除フラグが立つ場合と、それとは別の業務削除フラグなどが立つ場合がある。
    // 前者の場合、ほぼ同時に2画面から同一レコードを削除すると、後の処理になった方はこのselectの結果がnullとなるため拾う
    // ちなみに後者の場合はselect結果が取れるので、EDIT同様楽観的排他制御エラーになる
    if (pattern == DELETE && optional.isEmpty()) {
      String msg = "jp.ecuacion.splib.web.common.message.sameRecordAlreadyDeleted";
      throw new CustomizedValidationAppException(msg);
    }

    E e = optional.get();

    // findForUpdate == true の場合は、監査項目を削除しておく。
    // springの機能では、既に値が埋まっていると監査項目を更新してくれない・・・
    if (pattern == EDIT || pattern == DELETE) {
      jpaServiceUtil.putNullToAuditingFields(e);
    }

    // versionが異なる場合は楽観的排他制御エラーとする
    if (!versionInScreen.equals(getVersion(e).toString())) {
      throw new ObjectOptimisticLockingFailureException("some class", id);
    }

    return e;
  }

  //
  // getListForm
  //

  public abstract SplibListForm<? extends SplibRecord> getListForm(
      SplibSearchConditionForm searchConditionForm) throws MultipleAppException;

  protected abstract Specification<E> getSpecs(SplibSearchConditionForm searchConditionForm);

  protected Page<E> getListFormCommon(SplibSearchConditionForm searchConditionForm,
      JpaSpecificationExecutor<E> repository) {
    Specification<E> specs = getSpecs(searchConditionForm);

    // 指定条件での検索結果の全件数を取得。併せてページ調整
    searchConditionForm.setNumberOfRecordsAndAdjustCurrentPageNumger(repository.count(specs));

    // ここでは指定ページ分のレコードのみ取得
    return repository.findAll(specs, searchConditionForm.getPageRequest());
  }

  //
  // others
  //

  public abstract SplibEditForm getInsertForm(SplibEditForm editForm) throws AppException;

  /**
   * 引数のformは、listから選択された行のidとversionを受け取るためのformとしてたまたまeditFormを使用しているのみ。
   * そのidを元にレコード全体を取得したものを改めてeditFormとして戻す。
   */
  public abstract SplibEditForm getUpdateForm(SplibEditForm editForm) throws Exception;

  public abstract void edit(SplibEditForm editForm, UserDetails loginUser) throws Exception;

  public abstract void prepareEditForm(SplibEditForm editForm);

  public static enum ActionPattern {
    GET_UPDATE_FORM, EDIT, DELETE;
  }
}
