package jp.ecuacion.splib.web.service;

import jakarta.servlet.http.HttpServletRequest;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.lib.jpa.dbaccess.EntityManagerWrapper;
import jp.ecuacion.splib.core.container.DatetimeFormatParameters;
import jp.ecuacion.splib.core.form.record.SplibRecord;
import jp.ecuacion.splib.web.form.SplibListForm;
import jp.ecuacion.splib.web.form.SplibSearchForm;
import jp.ecuacion.splib.web.util.SplibUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.transaction.annotation.Transactional;

@Transactional(rollbackFor = Exception.class)
public abstract class SplibSearchListJpaService<E extends AbstractEntity>
    extends SplibSearchListService implements SplibJpaServiceInterface<E> {

  @Autowired
  private HttpServletRequest request;

  protected EntityManagerWrapper em;

  //
  // common
  //

  /**
   * offsetはlogin画面でのonload時に呼ばれるため、login画面を開いた状態で放置した場合は値がnullでエラーになる。
   */
  public DatetimeFormatParameters getParams() {
    return new SplibUtil().getParams(request);
  }

  /**
   * record内のlocalDate項目（String）をLocalDate形式で取得。
   */
  protected LocalDate localDate(String date) {
    return (date == null || date.equals("")) ? null
        : LocalDate.parse(date, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
  }

  public void setEntityManager(EntityManagerWrapper em) {
    this.em = em;
  }

  public abstract SplibListForm<? extends SplibRecord> getListForm(SplibSearchForm searchForm)
      throws MultipleAppException;

  protected abstract Specification<E> getSpecs(SplibSearchForm searchForm);

  protected Page<E> getListFormCommon(SplibSearchForm searchForm,
      JpaSpecificationExecutor<E> repository) {
    Specification<E> specs = getSpecs(searchForm);

    // 指定条件での検索結果の全件数を取得。併せてページ調整
    searchForm.setNumberOfRecordsAndAdjustCurrentPageNumger(repository.count(specs));

    // ここでは指定ページ分のレコードのみ取得
    return repository.findAll(specs, searchForm.getPageRequest());
  }

  //
  // others
  //

  public static enum ActionPattern {
    GET_UPDATE_FORM, EDIT, DELETE;
  }
}
