package jp.ecuacion.splib.web.service;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Locale;
import java.util.Set;
import jp.ecuacion.lib.core.exception.checked.AppWarningException;
import jp.ecuacion.splib.web.form.SplibGeneralForm;
import org.springframework.orm.ObjectOptimisticLockingFailureException;

public class SplibGeneralService {

  /* ----------------------- */
  /* warning */
  /* ----------------------- */

  protected void throwWarning(Set<String> confirmedWarningMessageSet, Locale locale, String msgId)
      throws AppWarningException {
    if (!confirmedWarningMessageSet.contains(msgId)) {
      throw new AppWarningException(locale, msgId);
    }
  }

  /**  */
  protected void throwWarning(Set<String> confirmedWarningMessageSet, Locale locale,
      String buttonName, String msgId, String... params) throws AppWarningException {
    if (!confirmedWarningMessageSet.contains(msgId)) {
      throw new AppWarningException(locale, buttonName, null, msgId, params);
    }
  }

  /* ----------------------- */
  /* exclusive lock by file */
  /* ----------------------- */

  /** lock fileの最終更新時刻を取得。楽観的排他制御を行うために使用。 */
  protected String getLockFileVersion(File lockFile) throws IOException {
    // ディレクトリが存在しなければ作成
    lockFile.getParentFile().mkdirs();

    // ファイルが存在しなければ作成
    if (!lockFile.exists()) {
      lockFile.createNewFile();
    }

    return Instant.ofEpochMilli(lockFile.lastModified()).atOffset(ZoneOffset.ofHours(0)).toString();
  }

  /**
   * lock fileによる悲観的排他制御を実施したい場合に使用できるutility method.
   * lockFileとして使用するpathを指定したFileオブジェクトを渡せば、あとは諸々やってくれる。 exclusiveLockActivatedByLockFile()
   * は、abstractにして実装必須にすると面倒（fileによるロックの頻度は高くない）なので、
   * 本クラスにて処理なしの通常メソッドとして実装しておき、使用する場合はそれをoverride、とする。
   */
  protected void fileLock(File lockFile, String version, SplibGeneralForm form) throws Exception {
    FileLock lockedObject = null;
    try (FileChannel channel =
        FileChannel.open(lockFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);) {

      // ファイルロックを試行。例外発生ではなくlockedObjectがnullの場合は、ロック取得に失敗したことになる場合もあるらしい
      lockedObject = channel.tryLock();
      if (lockedObject == null) {
        throw new ObjectOptimisticLockingFailureException((String) null, null);
      }

      // 画面表示時のtimestampとの差異比較
      String fileTimestamp = getLockFileVersion(lockFile);
      if (!version.equals(fileTimestamp)) {
        throw new ObjectOptimisticLockingFailureException((String) null, null);
      }

      exclusiveLockActivatedByLockFile(lockFile, form);

      // 特に使用はしないのだが、lockFileを更新する目的でtimestampの文字列を書き込んでおく。
      ByteBuffer src = ByteBuffer.allocate(30);
      byte[] bytes = LocalDateTime.now().toString().getBytes();
      src.put(bytes);
      src.position(0);

      channel.write(src);

      lockedObject.release();

    } catch (IOException | OverlappingFileLockException ex) {
      throw new ObjectOptimisticLockingFailureException((String) null, null);
    }
  }

  /**
   * file lockを使用する場合は本メソッドをoverride。 子クラスで実装必須になると面倒（使用しない場合の方が多いので）なことからabstractにはしない。
   */
  protected void exclusiveLockActivatedByLockFile(File lockFile, SplibGeneralForm form)
      throws Exception {

  }
}
