package jp.ecuacion.splib.web.controller.internal;

import jakarta.annotation.Nonnull;
import java.util.Set;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerData;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerNoJpa;
import jp.ecuacion.splib.web.controller.SplibBaseController;
import jp.ecuacion.splib.web.enums.LoginStateEnum;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibEditForm;
import jp.ecuacion.splib.web.form.SplibForm;
import jp.ecuacion.splib.web.util.TransactionTokenUtil;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;

public abstract class SplibBaseWithRecordController extends SplibBaseController {

  /**
   * form直下のrecordのfield名。対応するentity名を使用する。 特にlist-edit
   * templateでは、form配下には一つのみのrecordを保持するルールのため、entity名と同一にしておくのが推奨。 （list-edit
   * templateではそれ以外のパターンをや流必要性がなく未実施のためサポート外）
   * <p>
   * java側では、spring側ではrecordNameが必要だが毎回書くのが面倒な際に自動補完する目的などで使用。
   * また本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用。
   * </p>
   * <p>
   * layout-generalの場合は、recordが存在しない場合もある。その場合は""を指定。（nullは不可）
   * </p>
   */
  @Nonnull
  protected String rootRecordName;

  /** functionName, rootRecordNameを指定。 */
  public SplibBaseWithRecordController(LoginStateEnum loginState, String functionName,
      String rootRecordName) {
    super(loginState, functionName);
    this.rootRecordName = rootRecordName == null ? "" : rootRecordName;
  }

  public String getRootRecordName() {
    return rootRecordName;
  }

  /**
   * 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。
   * ちなみにmodel自体は、Controllerの保持する値に依存しないため、SplibControllerAdviceにてrequestに追加している。
   */
  @ModelAttribute
  private void setCommonParamsToModel(Model model) {
    model.addAttribute("rootRecordName", rootRecordName);
  }

  /**
   * 以下を実施。 設定項目の値に従ってpulldownの選択肢を動的に変更したい場合などは、敢えてvalidation checkをしたくない場面もあるので、 validation
   * checkの要否を持たせている。 - transactionToken check - validation check
   */
  protected void commonProc(SplibForm form, Model model, BindingResult result,
      boolean needsValidationCheck) throws InputValidationException, AppException {

    // transactionToken check
    String tokenFromHtml =
        (String) request.getParameter(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    @SuppressWarnings("unchecked")
    Set<String> tokenSet = (Set<String>) request.getSession()
        .getAttribute(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    if (tokenSet != null && tokenFromHtml != null) {
      if (!tokenSet.contains(tokenFromHtml)) {
        // tokenでのエラーは、edit画面でのエラーであってもlistに戻りたい（※）のでinfoからnextPageOnErrorを除去
        // （※）edti画面で新規登録 -> browser back -> 登録、とやると同じ内容のデータを2回登録できてしまうため。
        SplibExceptionHandlerData info = (SplibExceptionHandlerData) request
            .getAttribute(SplibExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING);
        info.setNextPageOnError(null);

        String msgId = "jp.ecuacion.splib.web.tool.common.message.tokenInvalidate";
        throw new CustomizedValidationAppException(msgId);
      }

      tokenSet.remove(tokenFromHtml);
    }

    if (needsValidationCheck) {
      // input validation
      boolean hasNotEmptyError = false;
      if (form instanceof SplibEditForm) {
        SplibEditForm f = (SplibEditForm) form;
        hasNotEmptyError = f.hasNotEmptyError();
      }

      if (hasNotEmptyError || (result != null && result.hasErrors())) {
        throw new InputValidationException(form);
      }
    }
  }

}
