package jp.ecuacion.splib.web.controller;

import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.splib.core.form.record.SplibRecord;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerData;
import jp.ecuacion.splib.web.advice.SplibExceptionHandlerNoJpa;
import jp.ecuacion.splib.web.controller.internal.SplibBaseWithRecordController;
import jp.ecuacion.splib.web.enums.LoginStateEnum;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.form.SplibEditForm;
import jp.ecuacion.splib.web.form.SplibForm;
import jp.ecuacion.splib.web.form.SplibListForm;
import jp.ecuacion.splib.web.form.SplibSearchConditionForm;
import jp.ecuacion.splib.web.service.SplibService;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibListEditController
      <S extends SplibSearchConditionForm, E extends SplibEditForm>
    extends SplibBaseWithRecordController {

  /** edit success時にredirectされる先を変更する場合に使用。通常は未設定で良い。 */
  private String editSuccessRedirectPath;

  protected abstract S getNewSearchConditionForm();

  public abstract <T extends AbstractEntity> SplibService<T> getService();

  public SplibListEditController(LoginStateEnum loginState, String functionName,
      String rootRecordName) {
    super(loginState, functionName, rootRecordName);
  }

  public SplibListEditController(LoginStateEnum loginState, String functionName) {
    super(loginState, functionName, functionName);
  }

  protected void setEditSuccessRedirectPath(String path) {
    this.editSuccessRedirectPath = path;
  }

  @GetMapping(value = "search")
  public String search(Model model, S f) throws MultipleAppException {
    S searchForm = getProperSearchConditionForm(model, f);

    SplibService<?> service = getService();
    service.setEntityManager(dbAccessManager.getEntityManager());

    SplibListForm<? extends SplibRecord> listForm = getService().getListForm(searchForm);
    listForm.setMenuName(searchForm.getMenuName());
    model.addAttribute(functionName + "ListForm", listForm);

    return functionName + "List";
  }

  /** list画面での削除時のエラー発生など、エラーがあって一覧画面に戻る場合に使用。 */
  public String search(Model model, String menuName) throws MultipleAppException {
    S form = getNewSearchConditionForm();
    form.setMenuName(menuName);

    return search(model, form);
  }

  @SuppressWarnings({"unchecked"})
  protected S getProperSearchConditionForm(Model model, S f) {

    String formName = functionName + "SearchConditionForm";
    String key = getSessionKey(formName, f);
    if (f != null) {
      // 検索画面上で検索された場合はその条件を採用し保管。
      // それ以外の場合（メニューのリンクなどから来た場合など）はmodelに設定されたものをそのまま使用。（ここでは何もしない）
      if (f.isRequestFromSearchForm()) {
        request.getSession().setAttribute(key, f);
      }

      // 初回アクセスでsessionに存在しない場合は引数のformを使用
      if (request.getSession().getAttribute(key) == null) {
        request.getSession().setAttribute(key, f);
      }

    } else {
      // その前に検索しているはずなので、引数のfがnullで、かつsessionにも情報がない、という場合はあり得ない。
      if (request.getSession().getAttribute(formName) == null) {
        throw new RuntimeException("f == null cannot be occurred.");
      }
    }

    S formUsedForSearch = (S) request.getSession().getAttribute(getSessionKey(formName, f));

    // 使用するsearchConditionformがspring mvcからも使用されるよう、modelにも入れておく
    model.addAttribute(formName, formUsedForSearch);

    return formUsedForSearch;
  }

  /** fがnullの場合null Pointerが発生するためメソッド冒頭で定義することはできず、別メソッドとした。 */
  private String getSessionKey(String formName, S f) {
    return formName + (f == null || f.getMenuName() == null || f.getMenuName().equals("") ? ""
        : "." + f.getMenuName());
  }

  @GetMapping(value = "search", params = "conditionClear")
  public String searchConditionClear(Model model, S f) throws MultipleAppException {
    // 情報をクリアするための設定を行う
    String formName = functionName + "SearchConditionForm";
    String sessionKey = formName
        + (f.getMenuName() == null || f.getMenuName().equals("") ? "" : "." + f.getMenuName());
    request.getSession().setAttribute(sessionKey, null);

    return "redirect:/" + loginState.getCode() + "/" + functionName + "/search?"
        + getUrlParams(f.getMenuName(), false);
  }

  @GetMapping(value = "list", params = "showInsertForm")
  public String showInsertForm(Model model, E f) throws AppException {
    prepareWithMovingToListOnError(f, model);

    SplibEditForm form = getService().getInsertForm(f);
    form.setIsInsert(true);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  @GetMapping(value = "list", params = "showUpdateForm")
  public String showUpdateForm(Model model, E f) throws Exception {
    prepareWithMovingToListOnError(f, model);

    SplibEditForm form = getService().getUpdateForm(f);
    form.setIsInsert(false);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  /**
   * 一覧からの削除処理。 delete用URLをお気に入りに入れられても困るので、本当はPostでの通信にしたいところだったが、
   * 呼び出し先を、showUpdateForm()と同じ「action」にしており、その状態でdeleteの場合のみpostに変更して投げてもspring mvcが判別つかないようで、
   * 「postは無効です」的なエラーが発生する。
   * deleteのみ別のformとするのもやりにくいし、formで持つactionをjavascriptで書き換えるのも微妙（th:actionでthymeleafが制御しているところなので）
   * なことから、GETとすることを許容する。以下の対策を行うことで、実質問題も起こらないと考えられる。 -
   * PRGを使うことで、delete中にシステムエラーでも起きない限り、delete時のURLがブラウザのURLバーには残らない
   */
  @GetMapping(value = "list", params = "delete")
  public String delete(Model model, E f, @AuthenticationPrincipal UserDetails loginUser)
      throws Exception {
    prepareWithMovingToListOnError(f, model);
    getService().delete(f, loginUser);

    return "redirect:/" + loginState.getCode() + "/" + functionName + "/search?success"
        + getUrlParams(f.getMenuName(), true);
  }

  protected String getUrlParams(String menuName, boolean needsAmpersand) {
    String additionalString = "";
    if (menuName != null && !menuName.equals("")) {
      additionalString = (needsAmpersand ? "&" : "") + "menuName=" + menuName;
    }

    return additionalString;
  }

  @PostMapping(value = "edit")
  public String edit(@Validated E f, BindingResult result, Model model,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(f, model, functionName + "Edit", result, true);
    getService().edit(f, loginUser);

    return "redirect:" + (editSuccessRedirectPath == null
        ? "/" + loginState.getCode() + "/" + functionName + "/search?success"
        : editSuccessRedirectPath) + getUrlParams(f.getMenuName(), true);
  }

  @PostMapping(value = "edit", params = "back")
  public String back(@Validated E f, BindingResult result, Model model) {
    return "redirect:/" + loginState.getCode() + "/" + functionName + "/search?"
        + getUrlParams(f.getMenuName(), false);
  }

  @PostMapping(value = "edit", params = "updateDropDown=true")
  public String updateOptions(E f, Model model) throws Exception {
    prepare(f, model, functionName + "Edit", null, false);
    model.addAttribute(functionName + "EditForm", f);
    return functionName + "Edit";
  }

  /**
   * 本メソッドは以下の処理を行う。 input validationは個別処理で書いても良いのだが、それも含めて1行でかけるものを用意した。
   * 
   * <p>
   * 1. AppExceptionに対するメッセージ出力をControllerの各処理で行うのではなく@ControllerAdviceで行うための準備
   * - @ControllerAdvice側でmodelをDIしても中身が空のため、controller側のものを保管しておく <br>
   * - エラー発生時の戻り先ページを設定しておく
   * 
   * 2. input validation
   * </p>
   * <p>
   * 本処理は、新規登録／編集画面のように、requestで登録内容がformに入っており、それをそのまま返せばよいパターンで使用可能。
   * その場合、エラーの場合はhtmlを指定して戻れば良いので引数には戻りページのStringを指定する
   * </p>
   */
  protected void prepare(SplibForm form, Model model, String nextPageOnError, BindingResult result,
      boolean needsValidationCheck) throws InputValidationException, AppException {

    // 1. preparation
    SplibExceptionHandlerData info = new SplibExceptionHandlerData(this, form, nextPageOnError);
    request.setAttribute(SplibExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING, info);

    // resultがnullでない場合はもちろんそうだが、その他チェックでAppExceptionが発生した場合も、再表示のために情報が必要となるので基本呼び出しておく
    // （本当は、エラーが発生した場合にのみexceptionHandlerで呼び出すのがより好ましいのだが若干煩雑になるので一旦は毎回実施）
    if (form instanceof SplibEditForm) {
      // editの際にsubmitではこない値（selectの選択肢一覧など）を再度設定する
      prepareEditForm((SplibEditForm) form);
    }

    commonProc(form, model, result, needsValidationCheck);
  }

  /**
   * formチェックがない場合は、input validationのないこちらを使用。
   */
  protected void prepare(SplibForm form, Model model, String nextPageOnError)
      throws InputValidationException, AppException {
    prepare(form, model, nextPageOnError, null, false);
  }

  protected void prepareEditForm(SplibEditForm form) {
    getService().prepareEditForm(form);
  }

  /**
   * 一覧で対象を指定し削除、など、list画面での処理の場合、formには削除対象のデータが入っているだけで一覧全体の情報は持っていない。
   * そのため画面を再表示するには、エラーメッセージを詰めたmodelを引き渡しつつ、list表示用の処理を呼び出し再度一覧情報を取得する流れとなる。
   * <p>
   * その場合、削除などの処理終了後にlist表示用の処理を呼び出すことになるのだが、reflectionを使用しそのメソッドを引数に渡してもらう、
   * というのも煩雑なので、一覧表示用のlistは「public String list(Model model)」と固定し、そこに戻る処理とする。
   * </p>
   */
  protected void prepareWithMovingToListOnError(BindingResult result, SplibForm form, Model model)
      throws InputValidationException, AppException {
    // 1. preparation
    SplibExceptionHandlerData info = new SplibExceptionHandlerData(this, form);
    request.setAttribute(SplibExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING, info);

    commonProc(form, model, result, false);
  }

  /**
   * formチェックがない場合は、input validationのないこちらを使用。
   */
  protected void prepareWithMovingToListOnError(SplibForm form, Model model)
      throws InputValidationException, AppException {
    prepareWithMovingToListOnError(null, form, model);
  }
}
