package jp.ecuacion.splib.web.form.record;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import jp.ecuacion.splib.web.bean.HtmlItem;
import jp.ecuacion.splib.web.bean.HtmlItemNumber;

public interface RecordInterface {

  HtmlItem[] getHtmlItems();

  default HtmlItem getHtmlItem(String itemName) {
    return Arrays.asList(getHtmlItems()).stream()
        .collect(Collectors.toMap(e -> e.getItemName(), e -> e)).get(itemName);
  }

  default boolean needsCommas(String itemName) {
    HtmlItem item = Arrays.asList(getHtmlItems()).stream()
        .collect(Collectors.toMap(e -> e.getItemName(), e -> e)).get(itemName);

    if (item == null || !(item instanceof HtmlItemNumber)) {
      return false;
    }

    HtmlItemNumber numItem = (HtmlItemNumber) item;
    return numItem.getNeedsCommas();
  }

  /**
   * labelFieldNameを返す。
   * HtmlItemに指定したfieldNameをitemNameに指定すると、field_name.propertiesで解決できる形のitemNameが取得できる。
   * <p>
   * HtmlItem上の(itemName, labelItemname)は、rootRecord配下の属性の場合("id", "name")のように属性名単体で定義されているが、
   * getLabelItemName()にはrootRecordNameの引数も渡しており、戻り値は"acc.accName"のようにそのままfield名として取得できる仕様としている。
   * </p>
   * <p>
   * relationがある場合は、("accGroup.id", "accGroup.name")のようにHtmlItem上設定される。
   * この場合、戻り値に"acc."を付加するとfield_names.propertiesで解決できないので"acc."の付加はしない。
   * </p>
   * <p>
   * relationがある場合を含めて、各componentのitemNameに指定する文字列は、必ずHtmlItemに指定したitemNameと同一となる。
   * </p>
   */
  default String getLabelItemName(String rootRecordname, String itemName) {
    HtmlItem[] htmlItems = getHtmlItems();

    // rootRecordNameがaccの場合に、例外処理でエラーメッセージに項目名を出す際の処理は、itemNameが「acc.accId」となってしまう。
    // この場合でも対応できるように、itemNameの頭がrootRecordName + '.'の場合はそれを取り除く処理を行う。
    if (itemName.startsWith(rootRecordname + ".")) {
      itemName = itemName.substring((rootRecordname + ".").length());
    }

    Map<String, String> labelNameMap =
        Arrays.asList(htmlItems).stream().collect(Collectors.toMap(e -> e.getItemName(),
            e -> e.getLabelItemName() == null ? e.getItemName() : e.getLabelItemName()));
    String labelItemName = labelNameMap.get(itemName);

    // htmlItems上で定義がない場合 /
    // htmlItems上でlabelItemNameが未定義の場合はlabelItemNameがnullになるが、その場合はもとのfieldNameの値を入れておく
    labelItemName = labelItemName == null ? itemName : labelItemName;

    // htmlItems上、関連を使用している場合はaccGroup.nameのようにfieldName自体にentityNameを含む場合があるので考慮
    return labelItemName.contains(".") ? labelItemName : rootRecordname + "." + labelItemName;
  }

  /**
   * htmlItemsについて、個別機能のlistと共通のlistをmergeさせるために使用する。
   * あくまでutilレベルなので個別処理にしても良いのだが、極力個別コードを減らしたいので本クラスに保持する。
   */
  default HtmlItem[] mergeHtmlItems(HtmlItem[] items1, HtmlItem[] items2) {
    List<HtmlItem> list = new ArrayList<>(Arrays.asList(items1));

    // common側と個別側で同一項目が定義されている場合はエラーとする
    List<String> item1Keys = Arrays.asList(items1).stream().map(e -> e.getItemName()).toList();

    for (String item2Key : Arrays.asList(items2).stream().map(e -> e.getItemName()).toList()) {
      if (item1Keys.contains(item2Key)) {
        throw new RuntimeException(
            "'itemName' of HtmlItem[] duplicated with commonHtmlItems. key: " + item2Key);
      }
    }

    list.addAll(Arrays.asList(items2));

    return list.toArray(new HtmlItem[list.size()]);
  }
}
