package jp.ecuacion.splib.web.controller;

import jakarta.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import jp.ecuacion.splib.jpa.entitymanager.DbAccessManagerWithSpringJpa;
import jp.ecuacion.splib.web.enums.LoginStateEnum;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomBooleanEditor;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.lang.Nullable;
import org.springframework.ui.Model;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;

public abstract class SplibBaseController {

  @Autowired
  protected HttpServletRequest request;

  /** ecuacion-lib独自のDBアクセスの場合に使用。 */
  @Autowired
  protected DbAccessManagerWithSpringJpa dbAccessManager;

  /**
   * その機能を表す名前。 form, recordなど各種クラスのprefix、urlの/xxx/searchなどの"xxx"部分などに使用。
   * 複数の機能で同一の機能名を持たせるのは不可。
   * <p>
   * "accGroup"のようにentity名と同一にするのが推奨。（実装上機能名及び各クラスにentity名が冠されることになりわかりやすいため）
   * ただし、実際問題、同一entityを主として使用するが、複数画面用意する必要がある場合などに名前を変える必要があるため、
   * "accGroupManagement"などentity名とは異なる名前も指定可能。 主要となるEntityは同一だが用途が異なるため画面を分けたい、などの場合には本機能名を分ける。
   * 異なるControllerで同一のfunctionNameを指定することは不可。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
   * </p>
   */
  protected String functionName;
  
  /*
   * loginStateを保持。
   * redirectをさせる際など、controller側でもこの値を把握しないとURL指定ができないため、controller側に持たせる。
   */
  protected LoginStateEnum loginState;
  
  public String getFunctionName() {
    return functionName;
  }
  
  public SplibBaseController(LoginStateEnum loginState, String functionName) {
    this.loginState = loginState;
    this.functionName = functionName;
  }

  /** 
   * 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。
   * ちなみにmodel自体は、Controllerの保持する値に依存しないため、SplibControllerAdviceにてrequestに追加している。
   */
  @ModelAttribute
  private void setCommonParamsToModel(Model model) {
    model.addAttribute("loginState", loginState.getCode());
    model.addAttribute("functionName", functionName);
  }

  /** submit時、request parameter内の各項目を一括変更してくれる処理。 */
  @InitBinder
  public void initBinder(WebDataBinder binder) {
    // 文字列項目の""をnullに変更。これにより、実際には数字だが””が来るなどを防げる。
    binder.registerCustomEditor(String.class, new StringTrimmerEditor(true));
    // Boolean項目で、offのときにnullでなくfalseとなるよう対応（redmine#323参照）
    binder.registerCustomEditor(Boolean.class, new NullSupportCustomBooleanEditor());
  }

  /** nullが設定されたらfalseに自動変換するためのPropertyEditor。 */
  static class NullSupportCustomBooleanEditor extends CustomBooleanEditor {
    public NullSupportCustomBooleanEditor() {
      super(false);
    }

    @Override
    public void setAsText(@Nullable String text) throws IllegalArgumentException {
      // checkboxでは以下は発生しないはずなのだが一応対処
      if (text == null || text.equals("") || text.equals("null")) {
        setValue(null);
        return;
      }

      text = text.trim();

      // redmine #323対応により、複数の値がカンマ区切りで場合がある。その場合はtrueで返すのだが、
      // #323の前提はonとoffがくることなので、そうでない場合はエラーとする。
      List<String> list = Arrays.asList(text.split(","));
      if (list.size() > 1) {
        if (list.size() > 2) {
          throw new RuntimeException("Unpresumable.");
        }

        Collections.sort(list);
        if (!(list.get(0).equalsIgnoreCase("OFF") && list.get(1).equalsIgnoreCase("ON"))) {
          throw new RuntimeException("Unpresumable.");
        }

        setValue(Boolean.TRUE);
        return;
      }

      super.setAsText(text);
    }
  }
}
