package jp.ecuacion.splib.web.tool.form;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import jp.ecuacion.splib.core.form.record.SplibRecord;

public abstract class SplibForm {

  /**
   * 一つの機能の実装を複数メニューで使い回す場合に使用するメニュー名。
   * 
   * 使い回しをしない通常機能は、放置しnull（実際にはhtml側でhiddenを持っているのでそれがsubmitされ空文字になる）のままでよい。
   * searchConditionFormはsessionにformを保持し検索条件を保持するが、複数メニューで使い回す場合は
   * そのメニュー別に検索条件を保持するのが望ましいため、このmenuNameもsession保存時のキーとする。
   * 
   * 現行設計ではmenuNameをrequestで持ち回る形を採用している。 sessionに持たせた方が実装が楽と思われる部分もありながら、navBarから他のメニューを選択された時には
   * sessionに保管されたmenuNameではなくrequestの方のmenuNameを採用する、あたりをうまく回す必要があり、
   * その意味ではsessionに持たせるよりrequestで持ち回る方がわかりやすい（menuNameが空ならシステムエラーになるので）ことから
   * まずはrequestで持ち回り、の実装方式としておく。 ※PRGを使用しGETでのアクセスを可としている中で、個々のURLをコピって使用しても使用可能とする意味でも、
   * 全てのURLにmenuNameパラメータがついているのが望ましい、という観点もあるので付記。
   */
  protected String menuName;

  /**
   * warningを返した際に、それに対してOKした場合は、OKしたという履歴を残さないとまた再度同じチェックに引っかかりワーニングを出してしまう。
   * それを防ぐため、一度OKしたwarningは、messageIdを本fieldに保管することで避ける。
   * 複数のwarnに対する情報を保持できるよう、本項目はcsv形式とする。（対応するhiddenにjavascript側で","とメッセージIDを追加）
   */
  protected String confirmedWarnings;

  public String getMenuName() {
    return menuName;
  }

  public void setMenuName(String menuName) {
    this.menuName = menuName;
  }

  /** 引数は、"acc"のような形で渡される想定。 */
  public SplibRecord get(String itemName) throws IllegalArgumentException,
      IllegalAccessException, NoSuchFieldException, SecurityException {
    return (SplibRecord) this.getClass().getField(itemName).get(this);
  }

  public Field getRootRecordField() {
    // form内に保持しているrecordを取得。
    // 複数functionで同一の画面を使う場合に一方のformを継承した場合、親classがprivateで持っているfieldを取得する必要があるため
    // ループを回しての捜索となっている
    Class<?> cls = this.getClass();
    List<Field> checkList = new ArrayList<>();
    while (true) {
      Field[] fields = cls.getDeclaredFields();

      // 正しくrecordが存在しているかのチェックとrecordの取得
      for (Field field : fields) {
        if (SplibRecord.class.isAssignableFrom(field.getType())) {
          checkList.add(field);
        }
      }
      
      if (checkList.size() == 0) {
        if (cls.getSuperclass() == SplibRecord.class) {
          throw new RuntimeException(
              "SplibRecord field not found in form: " + this.getClass().getName());
          
        } else {
          cls = cls.getSuperclass();
        }

      } else if (checkList.size() > 1) {
        throw new RuntimeException(
            "Multiple SplibRecord field found in form: " + this.getClass().getName());

      } else {
        return checkList.get(0);
      }
    }
  }

  /** formからrecordを取得する処理。 */
  public Object getRootRecord() {
    return getRootRecord(getRootRecordField());
  }

  /** getLabelItemName() ではfieldも別途使用するため、fieldの二回取得は無駄なのでfieldを引数にするメソッドも作っておく。 */
  protected Object getRootRecord(Field rootRecordField) {
    rootRecordField.setAccessible(true);

    Object rootRecord = null;

    try {
      rootRecord = (Object) rootRecordField.get(this);

    } catch (IllegalArgumentException | IllegalAccessException e) {
      throw new RuntimeException(e);
    }

    return rootRecord;
  }

  public String getConfirmedWarnings() {
    return confirmedWarnings;
  }

  public boolean containsConfirmedWarning(String messageId) {
    return getConfirmedWarningMessageSet().size() > 0;
  }

  public Set<String> getConfirmedWarningMessageSet() {
    Set<String> rtnSet = new HashSet<>();

    if (confirmedWarnings == null || confirmedWarnings.equals("")) {
      return rtnSet;
    }

    rtnSet.addAll(Arrays.asList(confirmedWarnings.split(",")));
    return rtnSet;
  }

  public void setConfirmedWarnings(String confirmedWarnings) {
    this.confirmedWarnings = confirmedWarnings;
  }
}
