package jp.ecuacion.splib.web.tool.config;


import java.util.Arrays;
import java.util.List;
import org.springframework.boot.autoconfigure.security.servlet.PathRequest;
import org.springframework.context.annotation.Bean;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.servlet.util.matcher.MvcRequestMatcher;
import org.springframework.web.servlet.handler.HandlerMappingIntrospector;

/**
 * WebSecurityConfigのtemplate。 defaultSuccessUrlだけはシステム個別で設定したくなるため、abstractクラスとする。
 */
public abstract class SplibWebToolSecurityConfig {

  protected PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

  /** ログイン成功時の遷移先を指定。 */
  protected abstract String getDefaultSuccessUrl();

  /** ログインが必要なurlにアクセスした際の遷移先。login画面の場合と、それ以前の説明画面の場合があるため個別設定可能とした。 */
  protected abstract String getUrlWithLoginNeededPageAccessed();

  protected abstract List<AuthorizationBean> getRoleInfo();

  protected abstract List<AuthorizationBean> getAuthorityInfo();

  @Bean
  PasswordEncoder passwordEncoder() {
    return passwordEncoder;
  }

  @Bean
  MvcRequestMatcher.Builder mvc(HandlerMappingIntrospector introspector) {
    return new MvcRequestMatcher.Builder(introspector);
  }

  @Bean
  public SecurityFilterChain filterChain(HttpSecurity http, MvcRequestMatcher.Builder mvc)
      throws Exception {
    http.httpBasic().disable();

    http.formLogin().loginPage(getUrlWithLoginNeededPageAccessed())
        .loginProcessingUrl("/public/authenticate").usernameParameter("username")
        .passwordParameter("password").defaultSuccessUrl(getDefaultSuccessUrl(), true)
        .failureUrl("/public/login?error");

     http.authorizeHttpRequests()
     .requestMatchers(PathRequest.toStaticResources().atCommonLocations()).permitAll()
     .requestMatchers(mvc.pattern("/public/**")).permitAll();

    if (getRoleInfo() != null) {
      for (AuthorizationBean bean : getRoleInfo()) {
        List<MvcRequestMatcher> list =
            Arrays.asList(bean.requestMatchers).stream().map(str -> mvc.pattern(str)).toList();
        http.authorizeHttpRequests()
            .requestMatchers(list.toArray(new MvcRequestMatcher[list.size()]))
            .hasAnyRole(bean.roleOrAuthority);
      }
    }

    if (getAuthorityInfo() != null) {
      for (AuthorizationBean bean : getAuthorityInfo()) {
        http.authorizeHttpRequests().requestMatchers(bean.requestMatchers)
            .hasAnyAuthority(bean.roleOrAuthority);
      }
    }

    http.authorizeHttpRequests().anyRequest().authenticated();

    http.logout().logoutUrl("/public/logout").logoutSuccessUrl("/public/login?logoutDone");

    http.exceptionHandling().accessDeniedPage("/public/login?accessDenied");

    return http.build();
  }

  public static class AuthorizationBean {
    private String[] requestMatchers;
    private String[] roleOrAuthority;

    public AuthorizationBean() {

    }

    public AuthorizationBean(String requestMatchers, String roleOrAuthority) {
      this.requestMatchers = new String[] {requestMatchers};
      this.roleOrAuthority = new String[] {roleOrAuthority};
    }

    public AuthorizationBean(String[] requestMatchers, String roleOrAuthority) {
      this.roleOrAuthority = new String[] {roleOrAuthority};
      this.requestMatchers = requestMatchers;
    }

    public AuthorizationBean(String[] requestMatchers, String[] roleOrAuthority) {
      this.roleOrAuthority = roleOrAuthority;
      this.requestMatchers = requestMatchers;
    }

    public String[] getRequestMatchers() {
      return requestMatchers;
    }

    public void setRequestMatchers(String[] requestMatchers) {
      this.requestMatchers = requestMatchers;
    }

    public String[] getRoleOrAuthority() {
      return roleOrAuthority;
    }

    public void setRoleOrAuthority(String[] roleOrAuthority) {
      this.roleOrAuthority = roleOrAuthority;
    }
  }
}
