package jp.ecuacion.splib.web.tool.service;

import static jp.ecuacion.splib.web.tool.service.SplibWebToolService.ActionPattern.DELETE;
import static jp.ecuacion.splib.web.tool.service.SplibWebToolService.ActionPattern.EDIT;
import jakarta.servlet.http.HttpServletRequest;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import java.util.Set;
import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.AppWarningException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.lib.jpa.dbaccess.EntityManagerWrapper;
import jp.ecuacion.splib.core.container.DatetimeFormatParameters;
import jp.ecuacion.splib.core.form.record.SplibJpaRecord;
import jp.ecuacion.splib.jpa.repository.SplibJpaRepository;
import jp.ecuacion.splib.web.tool.form.SplibWebToolEditForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolListForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolSearchConditionForm;
import jp.ecuacion.splib.web.tool.util.SplibWebToolUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.orm.ObjectOptimisticLockingFailureException;
import org.springframework.security.core.userdetails.UserDetails;

public abstract class SplibWebToolService<E extends AbstractEntity> {

  @Autowired
  private HttpServletRequest request;
  
  protected EntityManagerWrapper em;

  //
  // common
  //

  /**
   * offsetはlogin画面でのonload時に呼ばれるため、login画面を開いた状態で放置した場合は値がnullでエラーになる。
   */
  public DatetimeFormatParameters getParams() {
    return new SplibWebToolUtil().getParams(request);
  }
  
  /** edit時にinsertかupdateかを判別する方法。小さな処理だが共通化しておく。 */
  protected boolean isInsert(String id) {
    return id == null || id.equals("");
  }

  /**
   * record内のlocalDate項目（String）をLocalDate形式で取得。
   */
  protected LocalDate localDate(String date) {
    return (date == null || date.equals("")) ? null
        : LocalDate.parse(date, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
  }

  public void setEntityManager(EntityManagerWrapper em) {
    this.em = em;
  }

  //
  // common: optimistic locking関連
  //

  /** findAndOptimisticLockingCheck() にて使用。selectのためのrepositoryを取得。 */
  protected abstract SplibJpaRepository<E, Long> getRepositoryForOptimisticLocking();

  /** findAndOptimisticLockingCheck() にて使用。監査項目（最終更新者・最終更新日時）をnullに設定。 */
  protected abstract void putNullToAuditingFields(E e);

  /** findAndOptimisticLockingCheck() にて使用。楽観的排他制御のためのversionを取得。 */
  protected abstract Object getVersion(E e);

  /**
   * edit画面への遷移・編集時・削除時のそれぞれで、DBからentityを取得する処理。 （edit画面への遷移では変更画面で画面表示するため、編集時・削除時はDB更新するため）
   * 
   * 取得と同時に楽観的排他制御エラー・同時にどうレコード削除時の制
   */
  protected AbstractEntity findAndOptimisticLockingCheck(String id, String versionInScreen,
      ActionPattern pattern) throws AppException {
    Optional<E> optional = getRepositoryForOptimisticLocking().findById(Long.valueOf(id));

    // 削除機能には、削除フラグが立つ場合と、それとは別の業務削除フラグなどが立つ場合がある。
    // 前者の場合、ほぼ同時に2画面から同一レコードを削除すると、後の処理になった方はこのselectの結果がnullとなるため拾う
    // ちなみに後者の場合はselect結果が取れるので、EDIT同様楽観的排他制御エラーになる
    if (pattern == DELETE && optional.isEmpty()) {
      String msg = "jp.ecuacion.splib.web.tool.common.message.sameRecordAlreadyDeleted";
      throw new CustomizedValidationAppException(msg);
    }

    E e = optional.get();

    // findForUpdate == true の場合は、監査項目を削除しておく。
    // springの機能では、既に値が埋まっていると監査項目を更新してくれない・・・
    if (pattern == EDIT || pattern == DELETE) {
      putNullToAuditingFields(e);
    }

    // versionが異なる場合は楽観的排他制御エラーとする
    if (!versionInScreen.equals(getVersion(e).toString())) {
      throw new ObjectOptimisticLockingFailureException("some class", id);
    }

    return e;
  }

  //
  // getListForm
  //

  public abstract SplibWebToolListForm<? extends SplibJpaRecord> getListForm(
      SplibWebToolSearchConditionForm searchConditionForm) throws MultipleAppException;

  protected abstract Specification<E> getSpecs(SplibWebToolSearchConditionForm searchConditionForm);

  protected Page<E> getListFormCommon(SplibWebToolSearchConditionForm searchConditionForm,
      JpaSpecificationExecutor<E> repository) {
    Specification<E> specs = getSpecs(searchConditionForm);

    // 指定条件での検索結果の全件数を取得。併せてページ調整
    searchConditionForm.setNumberOfRecordsAndAdjustCurrentPageNumger(repository.count(specs));

    // ここでは指定ページ分のレコードのみ取得
    return repository.findAll(specs, searchConditionForm.getPageRequest());
  }

  protected void throwWarning(Set<String> confirmedWarningMessageSet, String msgId)
      throws AppWarningException {
    if (!confirmedWarningMessageSet.contains(msgId)) {
      throw new AppWarningException(msgId);
    }
  }

  //
  // others
  //

  public abstract SplibWebToolEditForm getInsertForm(SplibWebToolEditForm editForm) throws AppException;

  /** 
   * 引数のformは、listから選択された行のidとversionを受け取るためのformとしてたまたまeditFormを使用しているのみ。
   * そのidを元にレコード全体を取得したものを改めてeditFormとして戻す。 
   */
  public abstract SplibWebToolEditForm getUpdateForm(SplibWebToolEditForm editForm) throws Exception;

  public abstract void edit(SplibWebToolEditForm editForm, UserDetails loginUser) throws Exception;

  public abstract void delete(SplibWebToolEditForm editForm, UserDetails loginUser) throws Exception;

  public abstract void prepareEditForm(SplibWebToolEditForm editForm);

  public static enum ActionPattern {
    GET_UPDATE_FORM, EDIT, DELETE;
  }
}
