package jp.ecuacion.splib.web.tool.controller;

import jakarta.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import jp.ecuacion.lib.core.entity.AbstractEntity;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.splib.core.form.record.SplibJpaRecord;
import jp.ecuacion.splib.jpa.entrypoint.EntryPointWithSpringJpa;
import jp.ecuacion.splib.web.tool.advice.SplibWebToolExceptionHandlerData;
import jp.ecuacion.splib.web.tool.advice.SplibWebToolExceptionHandlerNoJpa;
import jp.ecuacion.splib.web.tool.exception.InputValidationException;
import jp.ecuacion.splib.web.tool.form.SplibWebToolEditForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolListForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolSearchConditionForm;
import jp.ecuacion.splib.web.tool.service.SplibWebToolService;
import jp.ecuacion.splib.web.tool.util.TransactionTokenUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomBooleanEditor;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.lang.Nullable;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibWebToolController<S extends SplibWebToolSearchConditionForm, E extends SplibWebToolEditForm>
     extends EntryPointWithSpringJpa {

  @Autowired
  protected HttpServletRequest request;

  /**
   * form直下のrecordのfield名。 java側では、spring側ではrecordNameが必要だが毎回書くのが面倒な際に自動補完する目的などで使用。 画面では、optimistic
   * locking用のversion fieldを持つentityなどで使用。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
   */
  private String rootRecordName;

  /**
   * その機能を表す名前。 form, recordなど各種クラスのprefix、urlの/xxx/searchなどの"xxx"部分などに使用。
   * 
   * "accGroup"のようにentityと同一にするのが推奨。（実装上各クラスにentity名が冠されることになりわかりやすいため）
   * 一方で、"accGroupManagement"などentity名とは異なる名前も指定可能。 主要となるEntityは同一だが用途が異なるため画面を分けたい、などの場合には本機能名を分ける。
   * 異なるControllerで同一のfunctionNameを指定することは不可。 本fieldの値は、html側のtemplateにも渡され、同一のパラメータ名で使用される。
   */
  private String functionName;

  /** edit success時にredirectされる先を変更する場合に使用。通常は未設定で良い。 */
  private String editSuccessRedirectPath;

  protected abstract S getNewSearchConditionForm();

  public abstract <T extends AbstractEntity> SplibWebToolService<T> getService();

  public SplibWebToolController(String functionName) {
    this.functionName = functionName;
    this.rootRecordName = functionName;
  }

  public SplibWebToolController(String functionName, String rootRecordName) {
    this.functionName = functionName;
    this.rootRecordName = rootRecordName;
  }

  public String getRootRecordName() {
    return rootRecordName;
  }

  public String getFunctionName() {
    return functionName;
  }

  protected void setEditSuccessRedirectPath(String path) {
    this.editSuccessRedirectPath = path;
  }

  /** submit時、request parameter内の各項目を一括変更してくれる処理。 */
  @InitBinder
  public void initBinder(WebDataBinder binder) {
    // 文字列項目の""をnullに変更。
    binder.registerCustomEditor(String.class, new StringTrimmerEditor(true));
    // Boolean項目で、offのときにnullでなくfalseとなるよう対応（redmine#323参照）
    binder.registerCustomEditor(Boolean.class, new NullSupportCustomBooleanEditor());
  }

  /** 全処理に共通のmodelAttributeはSplibWebToolControllerAdviceに設定しているが、本controllerを使用する場合に必要なものはここで定義。 */
  @ModelAttribute
  protected void setCommonParamsToModel(Model model) {
    model.addAttribute("functionName", functionName);
    model.addAttribute("rootRecordName", rootRecordName);
  }

  @GetMapping(value = "search")
  public String search(Model model, S f) throws MultipleAppException {
    S searchForm = getProperSearchConditionForm(model, f);

    SplibWebToolService<?> service = getService();
    service.setEntityManager(getDbAccessManager().getEntityManager());

    SplibWebToolListForm<? extends SplibJpaRecord> listForm = getService().getListForm(searchForm);
    listForm.setMenuName(searchForm.getMenuName());
    model.addAttribute(functionName + "ListForm", listForm);

    return functionName + "List";
  }

  /** list画面での削除時のエラー発生など、エラーがあって一覧画面に戻る場合に使用。 */
  public String search(Model model, String menuName) throws MultipleAppException {
    S form = getNewSearchConditionForm();
    form.setMenuName(menuName);

    return search(model, form);
  }

  @SuppressWarnings({"unchecked"})
  protected S getProperSearchConditionForm(Model model, S f) {

    String formName = functionName + "SearchConditionForm";
    String key = getSessionKey(formName, f);
    if (f != null) {
      // 検索画面上で検索された場合はその条件を採用し保管。
      // それ以外の場合（メニューのリンクなどから来た場合など）はmodelに設定されたものをそのまま使用。（ここでは何もしない）
      if (f.isRequestFromSearchForm()) {
        request.getSession().setAttribute(key, f);
      }

      // 初回アクセスでsessionに存在しない場合は引数のformを使用
      if (request.getSession().getAttribute(key) == null) {
        request.getSession().setAttribute(key, f);
      }

    } else {
      // その前に検索しているはずなので、引数のfがnullで、かつsessionにも情報がない、という場合はあり得ない。
      if (request.getSession().getAttribute(formName) == null) {
        throw new RuntimeException("f == null cannot be occurred.");
      }
    }

    S formUsedForSearch = (S) request.getSession().getAttribute(getSessionKey(formName, f));

    // 使用するsearchConditionformがspring mvcからも使用されるよう、modelにも入れておく
    model.addAttribute(formName, formUsedForSearch);

    return formUsedForSearch;
  }

  /** fがnullの場合null Pointerが発生するためメソッド冒頭で定義することはできず、別メソッドとした。 */
  private String getSessionKey(String formName, S f) {
    return formName + (f == null || f.getMenuName() == null || f.getMenuName().equals("") ? ""
        : "." + f.getMenuName());
  }

  @GetMapping(value = "search", params = "conditionClear")
  public String searchConditionClear(Model model, S f) throws MultipleAppException {
    // 情報をクリアするための設定を行う
    String formName = functionName + "SearchConditionForm";
    String sessionKey = formName
        + (f.getMenuName() == null || f.getMenuName().equals("") ? "" : "." + f.getMenuName());
    request.getSession().setAttribute(sessionKey, null);

    return "redirect:/" + functionName + "/search?" + getUrlParams(f.getMenuName(), false);
  }

  @GetMapping(value = "list", params = "showInsertForm")
  public String showInsertForm(Model model, E f) throws AppException {
    prepareWithMovingToListOnError(f, model);

    SplibWebToolEditForm form = getService().getInsertForm(f);
    form.setIsInsert(true);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  @GetMapping(value = "list", params = "showUpdateForm")
  public String showUpdateForm(Model model, E f) throws Exception {
    prepareWithMovingToListOnError(f, model);

    SplibWebToolEditForm form = getService().getUpdateForm(f);
    form.setIsInsert(false);
    form.setMenuName(f.getMenuName());
    model.addAttribute(functionName + "EditForm", form);
    return functionName + "Edit";
  }

  /**
   * 一覧からの削除処理。 delete用URLをお気に入りに入れられても困るので、本当はPostでの通信にしたいところだったが、
   * 呼び出し先を、showUpdateForm()と同じ「action」にしており、その状態でdeleteの場合のみpostに変更して投げてもspring mvcが判別つかないようで、
   * 「postは無効です」的なエラーが発生する。
   * deleteのみ別のformとするのもやりにくいし、formで持つactionをjavascriptで書き換えるのも微妙（th:actionでthymeleafが制御しているところなので）なことから、
   * GETとすることを許容する。以下の対策を行うことで、実質問題も起こらないと考えられる。
   * 
   * - PRGを使うことで、delete中にシステムエラーでも起きない限り、delete時のURLがブラウザのURLバーには残らない
   */
  @GetMapping(value = "list", params = "delete")
  public String delete(Model model, E f, @AuthenticationPrincipal UserDetails loginUser)
      throws Exception {
    prepareWithMovingToListOnError(f, model);
    getService().delete(f, loginUser);

    return "redirect:/" + functionName + "/search?success" + getUrlParams(f.getMenuName(), true);
  }

  protected String getUrlParams(String menuName, boolean needsAmpersand) {
    String additionalString = "";
    if (menuName != null && !menuName.equals("")) {
      additionalString = (needsAmpersand ? "&" : "") + "menuName=" + menuName;
    }

    return additionalString;
  }

  @PostMapping(value = "edit")
  public String edit(@Validated E f, BindingResult result, Model model,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(f, model, functionName + "Edit", result, true);
    getService().edit(f, loginUser);

    return "redirect:" + (editSuccessRedirectPath == null ? "/" + functionName + "/search?success"
        : editSuccessRedirectPath) + getUrlParams(f.getMenuName(), true);
  }

  @PostMapping(value = "edit", params = "back")
  public String back(@Validated E f, BindingResult result, Model model) {
    return "redirect:/" + functionName + "/search?" + getUrlParams(f.getMenuName(), false);
  }

  @PostMapping(value = "edit", params = "updateDropDown=true")
  public String updateOptions(E f, Model model) throws Exception {
    prepare(f, model, functionName + "Edit", null, false);
    model.addAttribute(functionName + "EditForm", f);
    return functionName + "Edit";
  }

  /**
   * 本メソッドは以下の処理を行う。 input validationは個別処理で書いても良いのだが、それも含めて1行でかけるものを用意した。
   * 
   * <p>
   * 1. AppExceptionに対するメッセージ出力をControllerの各処理で行うのではなく@ControllerAdviceで行うための準備
   * - @ControllerAdvice側でmodelをDIしても中身が空のため、controller側のものを保管しておく <br>
   * - エラー発生時の戻り先ページを設定しておく
   * 
   * 2. input validation
   * </p>
   * <p>
   * 本処理は、新規登録／編集画面のように、requestで登録内容がformに入っており、それをそのまま返せばよいパターンで使用可能。
   * その場合、エラーの場合はhtmlを指定して戻れば良いので引数には戻りページのStringを指定する
   * </p>
   */
  protected void prepare(SplibWebToolForm form, Model model, String nextPageOnError,
      BindingResult result, boolean needsValidationCheck)
      throws InputValidationException, AppException {

    // 1. preparation
    SplibWebToolExceptionHandlerData info =
        new SplibWebToolExceptionHandlerData(this, form, nextPageOnError);
    request.setAttribute(SplibWebToolExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING, info);

    // resultがnullでない場合はもちろんそうだが、その他チェックでAppExceptionが発生した場合も、再表示のために情報が必要となるので基本呼び出しておく
    // （本当は、エラーが発生した場合にのみexceptionHandlerで呼び出すのがより好ましいのだが若干煩雑になるので一旦は毎回実施）
    if (form instanceof SplibWebToolEditForm) {
      // editの際にsubmitではこない値（selectの選択肢一覧など）を再度設定する
      prepareEditForm((SplibWebToolEditForm) form);
    }

    commonProc(form, model, result, needsValidationCheck);
  }

  protected void prepareEditForm(SplibWebToolEditForm form) {
    getService().prepareEditForm(form);
  }

  /**
   * formチェックがない場合は、input validationのないこちらを使用。
   */
  protected void prepare(SplibWebToolForm form, Model model, String nextPageOnError)
      throws InputValidationException, AppException {
    prepare(form, model, nextPageOnError, null, false);
  }

  /**
   * 一覧で対象を指定し削除、など、list画面での処理の場合、formには削除対象のデータが入っているだけで一覧全体の情報は持っていない。
   * そのため画面を再表示するには、エラーメッセージを詰めたmodelを引き渡しつつ、list表示用の処理を呼び出し再度一覧情報を取得する流れとなる。
   * <p>
   * その場合、削除などの処理終了後にlist表示用の処理を呼び出すことになるのだが、reflectionを使用しそのメソッドを引数に渡してもらう、
   * というのも煩雑なので、一覧表示用のlistは「public String list(Model model)」と固定し、そこに戻る処理とする。
   * </p>
   */
  protected void prepareWithMovingToListOnError(BindingResult result, SplibWebToolForm form,
      Model model) throws InputValidationException, AppException {
    // 1. preparation
    SplibWebToolExceptionHandlerData info = new SplibWebToolExceptionHandlerData(this, form);
    request.setAttribute(SplibWebToolExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING, info);

    commonProc(form, model, result, false);
  }

  /**
   * formチェックがない場合は、input validationのないこちらを使用。
   */
  protected void prepareWithMovingToListOnError(SplibWebToolForm form, Model model)
      throws InputValidationException, AppException {
    prepareWithMovingToListOnError(null, form, model);
  }

  private void commonProc(SplibWebToolForm form, Model model, BindingResult result,
      boolean needsValidationCheck) throws InputValidationException, AppException {

    // transactionToken check
    String tokenFromHtml =
        (String) request.getParameter(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    @SuppressWarnings("unchecked")
    Set<String> tokenSet = (Set<String>) request.getSession()
        .getAttribute(TransactionTokenUtil.SESSION_KEY_TRANSACTION_TOKEN);

    if (tokenSet != null && tokenFromHtml != null) {
      if (!tokenSet.contains(tokenFromHtml)) {
        // tokenでのエラーは、edit画面でのエラーであってもlistに戻りたい（※）のでinfoからnextPageOnErrorを除去
        // （※）edti画面で新規登録 -> browser back -> 登録、とやると同じ内容のデータを2回登録できてしまうため。
        SplibWebToolExceptionHandlerData info = (SplibWebToolExceptionHandlerData) request
            .getAttribute(SplibWebToolExceptionHandlerNoJpa.INFO_FOR_ERROR_HANDLING);
        info.setNextPageOnError(null);

        String msgId = "jp.ecuacion.splib.web.tool.common.message.tokenInvalidate";
        throw new CustomizedValidationAppException(msgId);
      }

      tokenSet.remove(tokenFromHtml);
    }

    if (needsValidationCheck) {
      // input validation
      boolean hasNotEmptyError = false;
      if (form instanceof SplibWebToolEditForm) {
        SplibWebToolEditForm f = (SplibWebToolEditForm) form;
        hasNotEmptyError = f.hasNotEmptyError();
      }

      if (hasNotEmptyError || (result != null && result.hasErrors())) {
        throw new InputValidationException(form);
      }
    }
  }

  /** nullが設定されたらfalseに自動変換するためのPropertyEditor */
  static class NullSupportCustomBooleanEditor extends CustomBooleanEditor {
    public NullSupportCustomBooleanEditor() {
      super(false);
    }

    @Override
    public void setAsText(@Nullable String text) throws IllegalArgumentException {
      // checkboxでは以下は発生しないはずなのだが一応対処
      if (text == null || text.equals("") || text.equals("null")) {
        setValue(null);
        return;
      }

      text = text.trim();

      // redmine #323対応により、複数の値がカンマ区切りで場合がある。その場合はtrueで返すのだが、
      // #323の前提はonとoffがくることなので、そうでない場合はエラーとする。
      List<String> list = Arrays.asList(text.split(","));
      if (list.size() > 1) {
        if (list.size() > 2) {
          throw new RuntimeException("Unpresumable.");
        }

        Collections.sort(list);
        if (!(list.get(0).equalsIgnoreCase("OFF") && list.get(1).equalsIgnoreCase("ON"))) {
          throw new RuntimeException("Unpresumable.");
        }

        setValue(Boolean.TRUE);
        return;
      }

      super.setAsText(text);
    }
  }
}
