package jp.ecuacion.splib.web.tool.advice;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.validation.ConstraintViolation;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.checked.AppWarningException;
import jp.ecuacion.lib.core.exception.checked.CustomizedValidationAppException;
import jp.ecuacion.lib.core.exception.checked.MultipleAppException;
import jp.ecuacion.lib.core.exception.checked.SingleAppException;
import jp.ecuacion.lib.core.util.BeanValidationUtil;
import jp.ecuacion.lib.core.util.ExceptionUtil;
import jp.ecuacion.lib.core.util.LogUtil;
import jp.ecuacion.lib.core.util.PropertyFileUtil;
import jp.ecuacion.splib.web.tool.bean.RequestResultBean;
import jp.ecuacion.splib.web.tool.exception.InputValidationException;
import jp.ecuacion.splib.web.tool.form.SplibWebToolEditForm;
import jp.ecuacion.splib.web.tool.form.SplibWebToolSearchConditionForm;
import jp.ecuacion.splib.web.tool.form.record.RecordInterface;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatusCode;
import org.springframework.ui.Model;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.servlet.ModelAndView;

public abstract class SplibWebToolExceptionHandlerNoJpa {

  public static final String INFO_FOR_ERROR_HANDLING = "ecuacion.spring.mvc.infoForErrorHandling";

  private LogUtil util = new LogUtil();

  @Autowired
  HttpServletRequest request;
  @Autowired
  SplibWebToolExceptionHandlerActionOnThrowable actionOnThrowable;

  protected SplibWebToolExceptionHandlerData getInfo() {
    return (SplibWebToolExceptionHandlerData) request.getAttribute(INFO_FOR_ERROR_HANDLING);
  }

  /** Controller処理前にrequestに格納しておいたmodelを取得 */
  private Model getModel() {
    return (Model) request.getAttribute(SplibWebToolControllerAdvice.REQUEST_KEY_MODEL);
  }

  /** "{0}"がmessageに含まれる場合はitemNameで置き換える。 */
  private String addFieldName(String message, String itemName) {
    return addFieldNames(message, new String[] {itemName});
  }

  private String addFieldNames(String message, String[] itemNames) {
    if (message.contains("{0}")) {
      message = MessageFormat.format(message, getItemNames(itemNames));
    }

    return message;
  }

  private String getItemNames(String[] itemNames) {
    StringBuilder sb = new StringBuilder();
    final String prependParenthesis = PropertyFileUtil.getMsg(request.getLocale(),
        "jp.ecuacion.splib.web.tool.common.message.itemName.prependParenthesis");
    final String appendParenthesis = PropertyFileUtil.getMsg(request.getLocale(),
        "jp.ecuacion.splib.web.tool.common.message.itemName.appendParenthesis");
    final String separator = PropertyFileUtil.getMsg(request.getLocale(),
        "jp.ecuacion.splib.web.tool.common.message.itemName.separator");

    boolean is1stTime = true;
    for (String itemName : itemNames) {

      // itemNameがmessages.propertiesにあったらそれに置き換える
      if (PropertyFileUtil.hasFieldName(itemName)) {
        itemName = PropertyFileUtil.getFieldName(request.getLocale(), itemName);
      }

      if (is1stTime) {
        is1stTime = false;

      } else {
        sb.append(separator);
      }

      sb.append(prependParenthesis + itemName + appendParenthesis);
    }

    return sb.toString();
  }

  @ExceptionHandler({AppWarningException.class})
  public ModelAndView handleAppWarningException(AppWarningException exception)
      throws MultipleAppException {
    SplibWebToolExceptionHandlerData info = getInfo();
    RequestResultBean requestResult =
        ((RequestResultBean) getModel().getAttribute(RequestResultBean.key));

    requestResult.setWarnMessage(exception.getMessageId(), PropertyFileUtil
        .getMsg(request.getLocale(), exception.getMessageId(), exception.getMessageArgs()));

    return common(info);
  }

  @ExceptionHandler({AppException.class})
  public ModelAndView handleAppException(AppException exception) throws MultipleAppException {
    SplibWebToolExceptionHandlerData info = getInfo();
    RequestResultBean requestResult =
        ((RequestResultBean) getModel().getAttribute(RequestResultBean.key));

    // MultipleAppExceptionも考慮し例外を複数持つ
    List<SingleAppException> exList = new ArrayList<>();
    if (exception instanceof MultipleAppException) {
      for (SingleAppException appEx : ((MultipleAppException) exception).getList()) {
        exList.add(appEx);
      }

    } else {
      exList.add((SingleAppException) exception);
    }

    // exList内のexceptionを一つずつ処理
    for (SingleAppException saex : exList) {

      String[] fields = null;
      if (saex instanceof CustomizedValidationAppException) {
        CustomizedValidationAppException ex = (CustomizedValidationAppException) saex;
        fields = (ex.getFields() == null) ? new String[] {} : ex.getFields().getFields();

        // fieldsは、html側としてはrootRecordNameから始まる必要があるが、java側からすると毎回書くのは面倒なので、
        // field名のみを記載することを推奨とする。
        // その場合、htmlと整合を取るためにここでrootRecordNameを追加しておく
        List<String> modifiedFieldList = new ArrayList<>();
        Objects.requireNonNull(fields);
        String prefix = info.getController().getRootRecordName() + ".";
        for (String field : fields) {
          modifiedFieldList.add(field.startsWith(prefix) ? field : (prefix + field));
        }

        fields = modifiedFieldList.toArray(new String[modifiedFieldList.size()]);
      }

      for (String message : new ExceptionUtil().getAppExceptionMessageList(saex,
          request.getLocale())) {
        // messageは既にmessage.propertiesのメッセージを取得し、パラメータも埋めた状態だが、
        // それでも{0}が残っている場合はfieldsの値を元に項目名を埋める。
        message = addFieldNames(message, fields);
        requestResult.setErrorMessage(message, fields);
      }
    }

    return common(info);
  }
  //
  // /**
  // * 楽観的排他制御の処理。 画面表示〜ボタン押下の間にレコード更新された場合は、手動でチェックなので直接CustomizedValidationAppExceptionを投げても良いのだが、
  // * 複数sessionで同一レコードを同時更新した場合（service内の処理内でのselectからupdateの間に別sessionがselect〜updateを完了）は
  // * JPAが自動でObjectOptimisticLockingFailureExceptionを投げてくるので、それも同様に処理できるよう楽観的排他制御エラーは
  // * ObjectOptimisticLockingFailureExceptionで統一しておく。
  // *
  // * @throws MultipleAppException
  // */
  // @ExceptionHandler({ObjectOptimisticLockingFailureException.class})
  // public ModelAndView handleObjectOptimisticLockingFailureException(
  // ObjectOptimisticLockingFailureException exception) throws MultipleAppException {
  // // 通常のチェックエラー扱いとする
  // return handleAppException(new CustomizedValidationAppException(
  // "jp.ecuacion.splib.web.tool.common.message.optimisticLocking"));
  // }
  //
  // @ExceptionHandler({PessimisticLockingFailureException.class})
  // public ModelAndView handlePessimisticLockingFailureException(
  // PessimisticLockingFailureException exception) throws MultipleAppException {
  // // 通常のチェックエラー扱いとする
  // return handleAppException(new CustomizedValidationAppException(
  // "jp.ecuacion.splib.web.tool.common.message.pessimisticLocking"));
  // }

  /**
   * 本来はspring mvcのvalidationに任せれば良いのだが、以下の2点が気に入らず、springで持っているerror情報(*)を    *
   * 修正しようとしたが「unmodifiablelist」で修正できなかったため、やむなく個別の処理を作成。
   * 
   * 1.画面上部に複数項目のエラー情報をまとめて表示する場合、並び順が実行するたびに変わる
   * 
   * 2.@Sizeなどのvalidationで、blankを無視する設定になっていないため、
   * 値がblankで、かつblankがエラーのannotationが付加されている場合でも、@Sizeなどのvalidationが走ってしまう。
   * 
   * 尚、controller層でのinput validationが漏れた場合、JPAの仕様によりDBアクセス直前にもう一度validationが行われ、
   * そこでConstraintViolationExceptionが発生する。
   * そうなる場合は、事前のチェックが漏れた場合のみであり、また単体のConstraintViolationExceptionしか取得できず、 想定しているエラー表示方法（input
   * validation分はまとめて結果を表示）とは異なり正しくない。
   * そのため実装不備とみなしシステムエラーとする（＝本クラス上で特別扱いはせず、"handleThrowable"methodで拾う）。
   * 
   * I(*) このような形でspring mvcのerror情報の取得が可能。最後のprintlnではfieldとcode（validator名："Size"など）を取得している。
   * 
   * // bean validation標準の各種validatorが、""の場合でもSize validatorのエラーが表示されるなどイマイチ。 //
   * 空欄の場合には空欄のエラーのみを出したいので、同一項目で、NotEmptyと別のvalidatorが同時に存在する場合はNotEmpty以外を間引く。 //
   * spring標準のvalidatorは、model内に以下のようなkey名で格納されているのでkey名を前方一致で取得 //
   * org.springframework.validation.BindingResult.driveRecordEditForm String bindingResultKey =
   * getModel().asMap().keySet().stream() .filter(key ->
   * key.startsWith("org.springframework.validation.BindingResult"))
   * .collect(Collectors.toList()).get(0); BeanPropertyBindingResult bindingResult =
   * (BeanPropertyBindingResult) getModel().asMap().get(bindingResultKey); for (FieldError error :
   * bindingResult.getFieldErrors()) { System.out.println(error.getCode());
   * System.out.println(error.getField()); }
   * 
   * @throws MultipleAppException
   */
  @ExceptionHandler({InputValidationException.class})
  public ModelAndView handleInputValidationException(InputValidationException exception)
      throws MultipleAppException {
    SplibWebToolExceptionHandlerData info = getInfo();
    RequestResultBean requestResult =
        ((RequestResultBean) getModel().getAttribute(RequestResultBean.key));

    // spring mvcでのvalidation結果からは情報がうまく取れないので、改めてvalidationを行う
    List<ValidationErrorInfoBean> errorList =
        new BeanValidationUtil().validate(exception.getForm(), request.getLocale()).stream()
            .map(cv -> new ValidationErrorInfoBean(cv)).collect(Collectors.toList());

    // NotEmptyのチェック結果を追加
    if (exception.getForm() instanceof SplibWebToolEditForm) {
      SplibWebToolEditForm form = (SplibWebToolEditForm) exception.getForm();
      errorList
          .addAll(form.validateNotEmpty(request.getLocale()).stream().collect(Collectors.toList()));
    }

    // 並び順を指定。今後項目名指定することも想定するが、一旦は並び順が固定されれば満足なので単純に並べる
    errorList = errorList.stream().sorted(getComparator()).collect(Collectors.toList());

    // bean validation標準の各種validatorが、""の場合でもSize validatorのエラーが表示されるなどイマイチ。
    // 空欄の場合には空欄のエラーのみを出したいので、同一項目で、NotEmptyと別のvalidatorが同時に存在する場合はNotEmpty以外を間引く。
    removeDuplicatedValidators(errorList);

    // messageを設定。{0}を項目名で埋める、などはspring mvcの機能でもやっているが、それだと使いにくいので別途実施
    for (ValidationErrorInfoBean cv : errorList) {

      String message = cv.getMessage();
      String itemId = cv.getPropertyPath().toString();

      String itemName = itemId;

      itemName = ((RecordInterface) exception.getForm().getRootRecord())
          .getLabelItemName(exception.getForm().getRootRecordField().getName(), itemName);

      message = addFieldName(message, itemName);

      requestResult.setErrorMessage(message, itemId);
    }

    // splibWebToolEditFormの場合は、プルダウンのリスト情報など表示に必要な情報を再取得する必要がある
    if (exception.getForm() instanceof SplibWebToolEditForm) {
      info.getController().getService().prepareEditForm((SplibWebToolEditForm) exception.getForm());
    }

    return common(info);
  }

  /**
   * bean validation標準の各種validatorが、""の場合でもSize validatorのエラーが表示されるなどイマイチ。
   * 空欄の場合には空欄のエラーのみを出したいので、同一項目で、NotEmptyと別のvalidatorが同時に存在する場合はNotEmpty以外を間引く。
   */
  private void removeDuplicatedValidators(List<ValidationErrorInfoBean> cvList) {

    // 一度、field名をkey, valueをsetとしcvを複数格納するmapを作成し、そのkeyに2件以上validatorが存在しかつNotEmptyが存在する場合は
    // NotEmpty以外を間引く、というロジックとする
    Map<String, Set<ValidationErrorInfoBean>> duplicateCheckMap = new HashMap<>();

    // 後続処理のため、NotEmptyを保持するkeyを保持しておく
    Set<String> keySetWithNotEmpty = new HashSet<>();

    for (ValidationErrorInfoBean cv : cvList) {
      String key = cv.getPropertyPath().toString();
      if (duplicateCheckMap.get(key) == null) {
        duplicateCheckMap.put(key, new HashSet<>());
      }

      duplicateCheckMap.get(key).add(cv);

      // NotEmptyの場合はkeySetWithNotEmptyに追加
      if (isNotEmptyValidator(cv)) {
        keySetWithNotEmpty.add(key);
      }
    }

    // duplicateCheckMapで、keySetWithNotEmptyに含まれるkeyのvalueは、NotEmpty以外を取り除く
    for (String key : keySetWithNotEmpty) {
      for (ValidationErrorInfoBean cv : duplicateCheckMap.get(key)) {
        if (!isNotEmptyValidator(cv)) {
          cvList.remove(cv);
        }
      }
    }
  }

  private boolean isNotEmptyValidator(ValidationErrorInfoBean cv) {
    String validatorClass = cv.getValidatorClass();
    return validatorClass.endsWith("NotEmpty") || validatorClass.endsWith("NotEmptyIfValid");
  }

  private Comparator<ValidationErrorInfoBean> getComparator() {
    return new Comparator<>() {
      @Override
      public int compare(ValidationErrorInfoBean f1, ValidationErrorInfoBean f2) {
        // 項目名で比較
        int result = f1.getPropertyPath().toString().compareTo(f2.getPropertyPath().toString());
        if (result != 0) {
          return result;
        }

        // 項目名が同じ場合はmessageTemplate(実質はvalidator種別）で比較
        result = f1.getValidatorClass().compareTo(f2.getValidatorClass());
        if (result != 0) {
          return result;
        }

        // validator種別も同じ場合は、@Patternのみと考えられる。
        // Patternの場合はregxpにより並び順が固定されるのでそれを含む文字列で比較
        String s1 = f1.getAnnotationDescriptionString();
        String s2 = f2.getAnnotationDescriptionString();
        return s1.compareTo(s2);
      }
    };
  }

  @ExceptionHandler({HttpRequestMethodNotSupportedException.class})
  public String handleHttpRequestMethodNotSupportedException(
      HttpRequestMethodNotSupportedException exception, Model model)
      throws CustomizedValidationAppException {

    util.logErr(exception, request.getLocale());
    actionOnThrowable.execute(exception);

    return "redirect:/public/login?error";
  }

  /**
   * WebでConstraintViolationExceptionが直接上がってくる場面は、事前のチェックが漏れた場合であり、
   * この場合は単体のConstraintViolationExceptionしか取得できないこともあり正しくないため、実装不備とみなしここに入れることとする。
   */
  @ExceptionHandler({Throwable.class})
  public ModelAndView handleThrowable(Throwable exception) {

    // entityのvalidationでエラーとなった場合、別のexceptionで飛んでくるが、
    // cause()を掘るとConstraintViolationExceptionが出現するので処理可能とする
    Throwable internalEx = exception;
    while (true) {
      if (internalEx.getCause() != null) {
        internalEx = internalEx.getCause();

      } else {
        break;
      }
    }

    util.logErr(exception, request.getLocale());

    // 個別appの処理。mail送信など。
    actionOnThrowable.execute(exception);

    return new ModelAndView("error", getModel() == null ? null : getModel().asMap(),
        HttpStatusCode.valueOf(500));
  }

  private <S extends SplibWebToolSearchConditionForm> ModelAndView common(
      SplibWebToolExceptionHandlerData info) throws MultipleAppException {

    String returnPage = null;
    Model model = getModel();
    // 一覧画面での更新処理の場合は、一覧取得処理を呼び出し
    if (info.getNextPageOnError() == null) {
      returnPage = info.getController().search(model, info.getForm().getMenuName());

    } else {
      returnPage = info.getNextPageOnError();
    }

    return new ModelAndView(returnPage, model.asMap());
  }

  /**
   * NotEmptyの効率的な実装のために動的なNotEmptyの適用（BaseRecordに@NotEmptyを記載しておき、何らかの条件でそれをactiveにする形）を検討したが、
   * 結果不可であることがわかった。その場合、個別ロジックでnot emptyチェックを行右必要があるが、別途実施されたbean validationの結果と合わせる。 その際、実際のbean
   * validationの結果（ConstraintViolation）を自分で生成するのはなかなか辛いので、bean validationの結果も含め
   * 必要な項目を本beanに入れて後処理を共通化する
   */
  public static class ValidationErrorInfoBean {
    private String message;
    private String propertyPath;
    private String validatorClass;

    private String annotationDescriptionString;

    /** こちらはnotEmptyのロジックチェック側で使用 */
    public ValidationErrorInfoBean(String message, String propertyPath, String validatorClass) {
      this.message = message;
      this.propertyPath = propertyPath;
      this.validatorClass = validatorClass;

      // これは@Pattern用なので実質使用はしないのだが、nullだとcompareの際におかしくなると嫌なので空白にしておく
      annotationDescriptionString = "";
    }

    public ValidationErrorInfoBean(ConstraintViolation<?> cv) {
      this.message = cv.getMessage();
      this.propertyPath = cv.getPropertyPath().toString();
      this.validatorClass = cv.getConstraintDescriptor().getAnnotation().annotationType().getName();
      this.annotationDescriptionString = cv.getConstraintDescriptor().getAnnotation().toString();
    }

    public String getMessage() {
      return message;
    }

    public String getPropertyPath() {
      return propertyPath;
    }

    public String getValidatorClass() {
      return validatorClass;
    }

    public String getAnnotationDescriptionString() {
      return annotationDescriptionString;
    }
  }
}
