package jp.ecuacion.splib.web.controller;

import jakarta.servlet.http.HttpServletRequest;
import jp.ecuacion.splib.jpa.entrypoint.EntryPointWithSpringJpa;
import jp.ecuacion.splib.web.exception.InputValidationException;
import jp.ecuacion.splib.web.exceptionhandler.SpringMvcExceptionHandler;
import jp.ecuacion.splib.web.exceptionhandler.SpringMvcExceptionHandlerData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;

public abstract class SpringMvcController extends EntryPointWithSpringJpa {

  @Autowired
  protected HttpServletRequest request;

  /**
   * 本メソッドは以下の処理を行う。 input validationは個別処理で書いても良いのだが、それも含めて1行でかけるものを用意した。
   * 
   * <p>
   * 1. AppExceptionに対するメッセージ出力をControllerの各処理で行うのではなく@ControllerAdviceで行うための準備
   * - @ControllerAdvice側でmodelをDIしても中身が空のため、controller側のものを保管しておく <br>
   * - エラー発生時の戻り先ページを設定しておく
   * 
   * 2. input validation
   * 
   * 3. 処理成功時のメッセージを埋め込み（logic validationでエラーになった場合はこのメッセージが上書きされる）
   * </p>
   * 
   * <p>
   * 本処理は、新規登録／編集画面のように、requestで登録内容がformに入っており、それをそのまま返せばよいパターンで使用可能。
   * その場合、エラーの場合はhtmlを指定して戻れば良いので引数には戻りページのStringを指定する
   * </p>
   */
  protected void prepare(BindingResult result, Model model, String nextPageOnError)
      throws InputValidationException {
    // 1. preparation
    SpringMvcExceptionHandlerData info = new SpringMvcExceptionHandlerData(model, nextPageOnError);
    request.setAttribute(SpringMvcExceptionHandler.INFO_FOR_ERROR_HANDLING, info);

    // 2. input validation
    if (result != null && result.hasErrors()) {
      throw new InputValidationException();
    }

    // 3. success message
    addSuccessMessage(model);
  }

  /** formチェックがない場合は、input validationのないこちらを使用。 */
  protected void prepare(Model model, String nextPageOnError)
      throws InputValidationException {
    prepare(null, model, nextPageOnError);
  }

  /**
   * 一覧で対象を指定し削除、など、list画面での処理の場合、formには削除対象のデータが入っているだけで一覧全体の情報は持っていない。
   * そのため画面を再表示するには、エラーメッセージを詰めたmodelを引き渡しつつ、list表示用の処理を呼び出し再度一覧情報を取得する流れとなる。
   * <p>
   * その場合、削除などの処理終了後にlist表示用の処理を呼び出すことになるのだが、reflectionを使用しそのメソッドを引数に渡してもらう、
   * というのも煩雑なので、一覧表示用のlistは「public String list(Model model)」と固定し、そこに戻る処理とする。
   * </p>
   */
  protected void prepareWithMovingToListOnError(BindingResult result, Model model)
      throws InputValidationException {
    // 1. preparation
    SpringMvcExceptionHandlerData info = new SpringMvcExceptionHandlerData(model, this);
    request.setAttribute(SpringMvcExceptionHandler.INFO_FOR_ERROR_HANDLING, info);

    // 2. input validation
    if (result != null && result.hasErrors()) {
      throw new InputValidationException();
    }

    // 3. success message
    addSuccessMessage(model);
  }

  /** formチェックがない場合は、input validationのないこちらを使用。 */
  protected void prepareWithMovingToListOnError(Model model)
      throws InputValidationException {
    prepareWithMovingToListOnError(null, model);
  }

  /** 上記メソッドから呼び出す本メソッドを作成しておく。実装は任意なのでnullを返すだけのメソッドとしておく。 */
  public String list(Model model) {
    return null;
  }

  protected void addSuccessMessage(Model model) {
    model.addAttribute("message", "処理が正常に完了しました。");
  }
}
