package jp.ecuacion.splib.web.controller;

import jakarta.annotation.Nonnull;
import jp.ecuacion.splib.web.bean.RedirectUrlBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPageOnSuccessBean;
import jp.ecuacion.splib.web.bean.RedirectUrlPathBean;
import jp.ecuacion.splib.web.constant.SplibWebConstants;
import jp.ecuacion.splib.web.form.SplibEditForm;
import jp.ecuacion.splib.web.service.SplibEditService;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;

public abstract class SplibEditController<E extends SplibEditForm, S extends SplibEditService<E>>
    extends SplibGeneral1FormController<E, S> {

  private PageTemplatePatternEnum pageTemplatePattern;

  protected RedirectUrlPathBean redirectPathOnSuccess;

  public SplibEditController(PageTemplatePatternEnum pageTemplatePattern,
      @Nonnull String function) {
    this(pageTemplatePattern, function, new ControllerContext());
  }

  public SplibEditController(PageTemplatePatternEnum pageTemplatePattern, @Nonnull String function,
      ControllerContext settings) {
    super(function, settings.subFunction("edit"));

    this.pageTemplatePattern = pageTemplatePattern;
  }

  /** 処理成功時の表示画面のdefault。 */
  public String getDefaultSubFunctionOnSuccess() {
    return pageTemplatePattern == PageTemplatePatternEnum.SINGLE ? "edit" : "searchList";
  }

  @GetMapping(value = "page", params = "showInsertForm")
  public String showInsertPage(Model model, E editForm,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(model, loginUser, editForm);
    editForm.setIsInsert(true);
    getService().page(editForm, loginUser);
    return getFunction() + "Edit";
  }

  @GetMapping(value = "page", params = "showUpdateForm")
  public String showUpdatePage(Model model, E form,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(model, loginUser, form);

    getService().getUpdatePage(form, loginUser);
    form.setIsInsert(false);
    form.setDataKind(form.getDataKind());
    model.addAttribute(getFunction() + "EditForm", form);
    return getFunction() + "Edit";
  }

  @PostMapping(value = "action", params = "insertOrUpdate")
  public String edit(@Validated E editForm, BindingResult result, Model model,
      @AuthenticationPrincipal UserDetails loginUser) throws Exception {
    prepare(model, loginUser, editForm.validate(result));
    getService().edit(editForm, loginUser);

    RedirectUrlBean redirectBean =
        redirectPathOnSuccess == null ? new RedirectUrlPageOnSuccessBean() : redirectPathOnSuccess;
    redirectBean.putParam(SplibWebConstants.KEY_DATA_KIND, editForm.getDataKind());

    return getReturnStringOnSuccess(redirectBean);
  }

  @PostMapping(value = "action", params = "back")
  public String back(@Validated E editForm, BindingResult result, Model model) {

    return getReturnStringOnSuccess(new RedirectUrlPageOnSuccessBean("searchList", "page")
        .noSuccessMessage().putParam(SplibWebConstants.KEY_DATA_KIND, editForm.getDataKind()));
  }

  public static enum PageTemplatePatternEnum {
    SINGLE, PAIR_WITH_SEARCH_LIST
  }
}
