package jp.ecuacion.splib.web.controller;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

/** 
 * 処理をせず、htmlを表示するだけのcontrollerが必要な場合に使用する。
 * そのようなControllerが複数あると邪魔なのでそれらを集約する目的で使用。
 * ひとつのクラスの中に、本クラスの子クラスを複数内部クラスとして定義することで効率的に複数のページ移動を実現。
 * <p>
 * page=xxxとparameterで指定すれば、xxx.htmlを表示してくれるcontroller。セキュリティも考慮。
 * </p>
 * <p>
 * 権限で画面表示の制御を行う場合には本Controllerは使用不可かも。
 * （/public/showPage/page?page=xxx をsecurityConfigに設定すればいけるかもしれないが未確認）
 * </p>
 */
@Controller
@Scope("prototype")
@RequestMapping(value = {"/public/showPage", "/account/showPage", "/admin/showPage"})
public class ShowPageController extends SplibBaseController {
  
  /**
   * 基本はgetでの使用を想定。
   * Session Timeout状態で、ログインボタンや他のPOST系ボタンを押す場合、session timeoutで弾かれ
   * 本ページにredirectされるが、そのredirectもPOSTで行われるため本処理にPOSTで入ってくることがある。
   * それを考慮しget/post両方を受け取る設定にしておく。
   */
  @RequestMapping(value = "page", method = {RequestMethod.POST, RequestMethod.GET})
  public String page(Model model, @RequestParam("page") String page) {

    //no checkだと脆弱性をつかれる可能性があるので、使用可能文字は限定しておく。
    String expression = "^[a-zA-Z0-9_]*$";
    if (!page.matches(expression)) {
      // システムエラーにすると面倒なので、home的なページへのredirectとしておく。
      // /public/home/pageとしているが、存在しなければredirectされて適切なページに飛ぶはず・・
      return "redirect:/public/home/page";
    }

    return page;
  }
}
