package jp.ecuacion.splib.core.util;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class SplibEnumUtil {

  /**
   * optionsのパターンは以下のいずれか。どれか一つしか指定できない。 （"|"は区切り文字。複数のoptionを指定するのに","を区切り文字として使用しているので、別の区切り文字を使用）
   * - including=value1|value2                  ※複数指定可能。
   * - excluding=value1|value2                  ※複数指定可能。
   * - firstCharOfCodeEqualTo=3|4               ※複数指定可能。"3"はcodeの一文字目を表す 
   * - firstCharOfCodeLessThanOrEqualTo=3       ※"3"はcodeの一文字目を表す 
   * - firstCharOfCodeGreaterThanOrEqualTo=3    ※"3"はcodeの一文字目を表す
   */
  public List<String[]> getListForHtmlSelect(EnumClassInfo enumInfo, String optionsString) {
    optionsString = (optionsString == null) ? "" : optionsString;
    String[] options = optionsString.split(",");

    // optionKey, optionValueを格納したmapを作成
    Map<String, String> optionMap = new HashMap<>();
    for (String option : options) {
      String optionKey = option.split("=")[0];
      String optionValue = option.split("=").length == 1 ? null : option.split("=")[1];

      // ListForHtmlSelectOptionEnumに存在するkeyのみを対象とし、それをMapに格納
      if (ListForHtmlSelectOptionEnum.getNameSet().contains(optionKey)) {
        optionMap.put(optionKey, optionValue);
      }
    }

    List<String[]> rtnList = new ArrayList<>();

    if (optionMap.keySet().size() > 1) {
      // 複数のoptionを設定した場合はエラー
      throw new RuntimeException(
          "Multiple options cannot be set. (" + optionMap.keySet().toString() + ")");
    }

    if (optionMap.keySet().size() == 0) {
      // optionがないので全てを追加して終了
      enumInfo.getValueList().stream()
          .forEach(value -> rtnList.add(new String[] {value.getCode(), value.getLabel()}));
      return rtnList;
    }

    // 以下、optionMap.keySet().size() == 1の場合。
    String optionKey = optionMap.keySet().stream().toList().get(0);
    String optionValue = optionMap.get(optionKey);

    for (EnumValueInfo value : enumInfo.getValueList()) {
      if ((optionKey.equals("including")
              && Arrays.asList(optionValue.split("\\|")).contains(value.getName()))
          || (optionKey.equals("excluding")
              && !Arrays.asList(optionValue.split("\\|")).contains(value.getName()))
          || (optionKey.equals("firstCharOfCodeEqualTo")
              && Arrays.asList(optionValue.split("\\|")).contains(value.getCode().substring(0, 1)))
          || (optionKey.equals("firstCharOfCodeLessThanOrEqualTo") && value.getCode()
              .substring(0, 1).getBytes(StandardCharsets.US_ASCII)[0] <= optionValue
                  .getBytes(StandardCharsets.US_ASCII)[0])
          || (optionKey.equals("firstCharOfCodeGreaterThanOrEqualTo") && value.getCode()
              .substring(0, 1).getBytes(StandardCharsets.US_ASCII)[0] >= optionValue
                  .getBytes(StandardCharsets.US_ASCII)[0])) {
        
        rtnList.add(new String[] {value.getCode(), value.getLabel()});
      }
    }

    return rtnList;
  }

  private static enum ListForHtmlSelectOptionEnum {
    including, excluding, firstCharOfCodeEqualTo, 
    firstCharOfCodeLessThanOrEqualTo, firstCharOfCodeGreaterThanOrEqualTo;

    public static Set<String> getNameSet() {
      Set<String> rtnSet = new HashSet<>();
      for (ListForHtmlSelectOptionEnum anEnum : ListForHtmlSelectOptionEnum.values()) {
        rtnSet.add(anEnum.name());
      }

      return rtnSet;
    }
  }

  public static class EnumClassInfo {
    private String enumClassName;
    private List<EnumValueInfo> valueList;

    public EnumClassInfo(String enumClassName, List<EnumValueInfo> valueList) {
      this.enumClassName = enumClassName;
      this.valueList = valueList;
    }

    public String getEnumClassName() {
      return enumClassName;
    }

    public void setEnumClassName(String enumClassName) {
      this.enumClassName = enumClassName;
    }

    public List<EnumValueInfo> getValueList() {
      return valueList;
    }

    public void setValueList(List<EnumValueInfo> valueList) {
      this.valueList = valueList;
    }
  }

  /**  */
  public static class EnumValueInfo {
    private String name;
    private String code;
    private String label;
    private String enumClassName;

    /**
     * EnumClassInfo 配下のListに保持される場合はこちらのconstructorを使用。
     * 本クラスは、EnumClassInfo配下に保持されそこでenumClassNameを保持しているため、本クラスのenumClassName はnullで実用上問題なし。
     */
    public EnumValueInfo(String name, String code, String label) {
      this.name = name;
      this.code = code;
      this.label = label;
    }

    /** EnumValueInfo を単体で取得する場合はこちらのconstructorを使用。 */
    public EnumValueInfo(String name, String code, String label, String enumClassName) {
      this(name, code, label);
      this.enumClassName = enumClassName;
    }

    public String getName() {
      return name;
    }

    public String getCode() {
      return code;
    }

    public String getLabel() {
      return label;
    }

    public String getEnumClassName() {
      return enumClassName;
    }
  }
}
