package jp.ecuacion.splib.batch.exceptionhandler;

import java.text.MessageFormat;
import java.util.List;
import java.util.Locale;
import jp.ecuacion.lib.core.exception.checked.AppException;
import jp.ecuacion.lib.core.exception.unchecked.RuntimeAppException;
import jp.ecuacion.lib.core.logging.DetailLogger;
import jp.ecuacion.lib.core.util.ExceptionUtil;
import jp.ecuacion.lib.core.util.LogUtil;
import jp.ecuacion.splib.batch.advice.SplibBatchAdvice;
import jp.ecuacion.splib.core.exceptionhandler.SplibExceptionHandlerAction;
import org.springframework.batch.repeat.RepeatContext;
import org.springframework.batch.repeat.exception.ExceptionHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * webでは汎用性を鑑みExceptionHandlerの使用有無を選択可能としていた（splib内のclassには@Componentはつけていない）が、
 * batchではそこまでの汎用性は不要と思われるので必ず使用する仕様とする。
 */
@Component
public class SplibExceptionHandler implements ExceptionHandler {

  @Autowired(required = false)
  private SplibExceptionHandlerAction action;

  private LogUtil logUtil = new LogUtil(this);
  private ExceptionUtil exUtil = new ExceptionUtil();
  
  private DetailLogger detailLog = new DetailLogger(this);

  private static final String NOT_FOUND_MSG_TMPL =
      "(The classname of {0} is null because the error occured outside {0}"
          + "or the advice to register current {0} is not implemented.)";

  @Override
  public void handleException(RepeatContext context, Throwable throwable) throws Throwable {

    StringBuilder sb = new StringBuilder();
    sb.append(formatMsg("job", SplibBatchAdvice.getCurrentJob(), true));
    sb.append(formatMsg("step", SplibBatchAdvice.getCurrentStep(), false));
    sb.append(formatMsg("tasklet or chunk", SplibBatchAdvice.getCurrentTaskletOrChunk(), false));

    logUtil.logErr(throwable, sb.toString());

    if (action != null) {
      try {
        action.execute(throwable);

      } catch (Throwable th) {
        logUtil.logErr(th);
      }
    }

    // appExceptionの場合、特にMultipleAppExceptionで複数のメッセージがある場合、
    // 一覧で見えないとわかりにくいので改めて一覧表示しておく。
    if (throwable instanceof AppException || throwable instanceof RuntimeAppException) {
      AppException appEx = null;
      if (throwable instanceof RuntimeAppException) {
        appEx = ((AppException) ((RuntimeAppException) throwable).getCause());

      } else {
        appEx = (AppException) throwable;
      }

      List<String> msgList = exUtil.getAppExceptionMessageList(appEx, Locale.getDefault());
      detailLog.info("==========");
      detailLog.info("[AppException メッセージ一覧]");
      msgList.stream().forEach(msg -> detailLog.info(msg));
      detailLog.info("==========");
    }

    throw throwable;
  }

  private String formatMsg(String kind, String name, boolean is1st) {
    return (is1st ? "" : ", ")
        + (name == null ? MessageFormat.format(NOT_FOUND_MSG_TMPL, "'" + kind + "'")
            : kind + ": " + name);
  }
}
