/*
 * Copyright © 2012 ecuacion.jp (info@ecuacion.jp)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.ecuacion.lib.jpa.entity;

import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.Id;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import jp.ecuacion.lib.core.annotation.RequireNonnull;
import org.apache.commons.lang3.StringUtils;

/**
 * Provides the customized jpa entity.
 */
public abstract class EclibEntity {

  /**
   * Returns the value of the designated fieldName.
   * 
   * @param fieldName fieldName. Cannot be {@code null}.
   * 
   * @return value. May be null when the value is null.
   */
  @Nullable
  public Object getValue(@RequireNonnull String fieldName) {
    try {
      Method m = this.getClass().getMethod("get" + StringUtils.capitalize(fieldName));
      Object value = m.invoke(this);

      return value;

    } catch (SecurityException | IllegalArgumentException
        | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
      throw new RuntimeException(e);
    }
  }

  /** 
   * Returns the surrogate-key field name, to which '@Id' is added.
   * 
   * @return surrogate key field name.
   */
  @Nonnull
  public String getPkFieldName() {
    Field[] fields = this.getClass().getDeclaredFields();

    for (Field field : fields) {
      Annotation annotation = field.getAnnotation(Id.class);
      if (annotation != null) {
        return Objects.requireNonNull(field.getName());
      }
    }

    // surrogate keyをつけることはecuacion-lib-jpaでは必須要件なのでRuntimeExceptionとする
    throw new RuntimeException("Field with '@Id' not found.");
  }

  /** 
   * Returns the auto-increment field name, to which '@GeneratedValue' is added.
   * 
   * @return auto-increment field name.
   */
  @Nonnull
  public Set<String> getAutoIncrementFieldNameSet() {
    Field[] fields = this.getClass().getDeclaredFields();
    Set<String> rtnSet = new HashSet<>();

    for (Field field : fields) {
      Annotation annotation = field.getAnnotation(GeneratedValue.class);
      if (annotation != null) {
        rtnSet.add(field.getName());
      }
    }

    return rtnSet;
  }

  /**
   * Returns if the designated field is auto-increment.
   * 
   * @param fieldName fieldName.
   * @return field is auto-increment
   */
  public boolean isAutoIncrement(@RequireNonnull String fieldName) {
    return getAutoIncrementFieldNameSet().contains(fieldName);
  }

  /**
   * Returns an array of fields which constructs an unique 
   * Constraint connected to natural key.
   * 
   * @return set of unique constraint column list.
   */
  @Nonnull
  public Set<List<String>> getSetOfUniqueConstraintFieldList() {
    Set<List<String>> rtnSet = new HashSet<>();

    Table table = this.getClass().getAnnotation(Table.class);
    UniqueConstraint[] ucs = table.uniqueConstraints();

    if (ucs == null) {
      return rtnSet;
    }

    for (UniqueConstraint uc : ucs) {
      rtnSet.add(Arrays.asList(uc.columnNames()));
    }

    return rtnSet;
  }

  /**
   * Returns if the entity has natural keys.
   * 
   * @return has natural keys.
   */
  public boolean hasNaturalKey() {
    return getSetOfUniqueConstraintFieldList().size() != 0;
  }

  /**
   * Provides preInsert procedure.
   * 
   * <p>When you use spring framework, this won't be used.</p>
   */
  public abstract void preInsert();

  /**
   * Provides preUpdate procedure.
   * 
   * <p>When you use spring framework, this won't be used.</p>
   */
  public abstract void preUpdate();

  /**
   * Returns if the entity has soft-delete field.
   * 
   * @return has soft-delete field.
   */
  public abstract boolean hasSoftDeleteField();

  /**
   * Returns field name array.
   * 
   * @return field name array.
   */
  public abstract String[] getFieldNameArr();
}
